#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
螺纹标准数据模块：提供各种标准螺纹的数据
包括公制、英制、管螺纹等标准规格和参数
"""

class ThreadStandards:
    """螺纹标准数据类，提供标准螺纹的尺寸数据"""
    
    # 标准螺纹直径常量 - 公制标准
    STANDARD_METRIC_DIAMETERS = [
        1.6, 2.0, 2.5, 3.0, 4.0, 5.0, 6.0, 8.0, 10.0, 12.0, 14.0, 16.0, 18.0, 20.0, 22.0, 24.0, 
        27.0, 30.0, 33.0, 36.0, 39.0, 42.0, 45.0, 48.0, 52.0, 56.0, 60.0, 64.0, 72.0, 80.0, 90.0, 100.0
    ]
    
    # 英制标准螺纹直径 (inch转换为mm)
    STANDARD_INCH_DIAMETERS = [
        # UNC/UNF常用规格转换为mm
        3.175,  # 1/8"
        4.7625, # 3/16"
        6.35,   # 1/4"
        7.9375, # 5/16"
        9.525,  # 3/8"
        11.1125, # 7/16"
        12.7,    # 1/2"
        15.875,  # 5/8"
        19.05,   # 3/4"
        22.225,  # 7/8"
        25.4     # 1"
    ]
    
    # 管螺纹标准直径 (G系列)
    STANDARD_PIPE_DIAMETERS = [
        # G系列管螺纹直径
        13.157,  # G1/4
        16.662,  # G3/8
        20.955,  # G1/2
        26.441,  # G3/4
        33.249,  # G1
        41.910,  # G1 1/4
        47.803,  # G1 1/2
        59.614   # G2
    ]
    
    # 标准螺距映射 - 公制标准
    METRIC_PITCH_MAP = {
        1.6: 0.35, 2.0: 0.4, 2.5: 0.45, 3.0: 0.5, 4.0: 0.7, 5.0: 0.8, 6.0: 1.0, 8.0: 1.25, 10.0: 1.5,
        12.0: 1.75, 14.0: 2.0, 16.0: 2.0, 18.0: 2.5, 20.0: 2.5, 22.0: 2.5,
        24.0: 3.0, 27.0: 3.0, 30.0: 3.5, 33.0: 3.5, 36.0: 4.0, 39.0: 4.0,
        42.0: 4.5, 45.0: 4.5, 48.0: 5.0, 52.0: 5.0, 56.0: 5.5, 60.0: 5.5, 64.0: 6.0,
        72.0: 6.0, 80.0: 6.0, 90.0: 6.0, 100.0: 6.0
    }
    
    # 英制标准螺距映射 (数值为每英寸螺纹数转换为mm的螺距)
    INCH_PITCH_MAP = {
        3.175: 0.794,   # 1/8" UNC (32 tpi)
        4.7625: 1.058,  # 3/16" UNC (24 tpi)
        6.35: 1.058,    # 1/4" UNC (20 tpi)
        7.9375: 1.270,  # 5/16" UNC (18 tpi)
        9.525: 1.411,   # 3/8" UNC (16 tpi)
        11.1125: 1.411, # 7/16" UNC (14 tpi)
        12.7: 1.693,    # 1/2" UNC (13 tpi)
        15.875: 1.814,  # 5/8" UNC (11 tpi)
        19.05: 2.117,   # 3/4" UNC (10 tpi)
        22.225: 2.117,  # 7/8" UNC (9 tpi)
        25.4: 2.540     # 1" UNC (8 tpi)
    }
    
    # 管螺纹螺距映射
    PIPE_PITCH_MAP = {
        13.157: 1.337,  # G1/4 (19 tpi)
        16.662: 1.337,  # G3/8 (19 tpi)
        20.955: 1.814,  # G1/2 (14 tpi)
        26.441: 1.814,  # G3/4 (14 tpi)
        33.249: 2.309,  # G1 (11 tpi)
        41.910: 2.309,  # G1 1/4 (11 tpi)
        47.803: 2.309,  # G1 1/2 (11 tpi)
        59.614: 2.309   # G2 (11 tpi)
    }
    
    # 螺纹标准角度 (度)
    THREAD_ANGLES = {
        'M': 60.0,    # 公制标准
        'UNC': 60.0,  # 统一粗牙
        'UNF': 60.0,  # 统一细牙
        'G': 55.0,    # 管螺纹
        'NPT': 60.0,  # 美国锥管螺纹
        'Tr': 30.0,   # 梯形螺纹
        'Acme': 29.0  # 美制梯形螺纹
    }
    
    # 英制螺纹直径到对应分数/尺寸的映射
    INCH_SIZE_MAP = {
        3.175: "1/8",
        4.7625: "3/16",
        6.35: "1/4",
        7.9375: "5/16",
        9.525: "3/8",
        11.1125: "7/16",
        12.7: "1/2",
        15.875: "5/8",
        19.05: "3/4",
        22.225: "7/8",
        25.4: "1"
    }
    
    # 管螺纹直径到尺寸的映射
    PIPE_SIZE_MAP = {
        13.157: "1/4",
        16.662: "3/8",
        20.955: "1/2",
        26.441: "3/4",
        33.249: "1",
        41.910: "1 1/4",
        47.803: "1 1/2",
        59.614: "2"
    }
    
    # 不同标准的螺纹代号前缀
    THREAD_PREFIXES = {
        "公制": "M",
        "英制_UNC": "",  # 英制UNC不需要前缀
        "英制_UNF": "UNF",
        "管螺纹": "G",
        "锥管螺纹": "NPT",
        "梯形": "Tr",
        "美制梯形": "Acme"
    }
    
    @staticmethod
    def get_thread_standard_info(diameter, thread_type=None):
        """
        根据螺纹直径和类型获取标准信息
        
        Args:
            diameter: 螺纹外径
            thread_type: 螺纹类型 (公制/英制/管螺纹)
            
        Returns:
            dict: 螺纹标准信息
        """
        info = {
            'standard_diameter': 0,
            'thread_type': '未知',
            'pitch': 0,
            'angle': 60.0,
            'prefix': ''
        }
        
        # 如果已知类型，只在该类型中查找
        if thread_type:
            if thread_type == "公制":
                std_diameters = ThreadStandards.STANDARD_METRIC_DIAMETERS
                pitch_map = ThreadStandards.METRIC_PITCH_MAP
                info['thread_type'] = "公制"
                info['prefix'] = "M"
                info['angle'] = 60.0
            elif thread_type == "英制":
                std_diameters = ThreadStandards.STANDARD_INCH_DIAMETERS
                pitch_map = ThreadStandards.INCH_PITCH_MAP
                info['thread_type'] = "英制"
                info['prefix'] = ""
                info['angle'] = 60.0
            elif thread_type == "管螺纹":
                std_diameters = ThreadStandards.STANDARD_PIPE_DIAMETERS
                pitch_map = ThreadStandards.PIPE_PITCH_MAP
                info['thread_type'] = "管螺纹"
                info['prefix'] = "G"
                info['angle'] = 55.0
            else:
                return info
            
            # 查找最接近的标准直径
            closest_diameter = min(std_diameters, key=lambda x: abs(x - diameter))
            if abs(closest_diameter - diameter) < 0.3:  # 允许一定误差
                info['standard_diameter'] = closest_diameter
                if closest_diameter in pitch_map:
                    info['pitch'] = pitch_map[closest_diameter]
                
        else:
            # 未知类型，查找所有标准
            all_standards = [
                ("公制", ThreadStandards.STANDARD_METRIC_DIAMETERS, ThreadStandards.METRIC_PITCH_MAP, 60.0, "M"),
                ("英制", ThreadStandards.STANDARD_INCH_DIAMETERS, ThreadStandards.INCH_PITCH_MAP, 60.0, ""),
                ("管螺纹", ThreadStandards.STANDARD_PIPE_DIAMETERS, ThreadStandards.PIPE_PITCH_MAP, 55.0, "G")
            ]
            
            best_match = None
            min_diff = float('inf')
            
            for std_name, std_diameters, pitch_map, angle, prefix in all_standards:
                for std_diameter in std_diameters:
                    diff = abs(std_diameter - diameter)
                    if diff < min_diff and diff < 0.3:
                        min_diff = diff
                        best_match = (std_name, std_diameter, pitch_map.get(std_diameter, 0), angle, prefix)
            
            if best_match:
                info['thread_type'] = best_match[0]
                info['standard_diameter'] = best_match[1]
                info['pitch'] = best_match[2]
                info['angle'] = best_match[3]
                info['prefix'] = best_match[4]
        
        return info


if __name__ == "__main__":
    # 测试代码
    import logging
    logging.basicConfig(level=logging.INFO)
    
    # 测试标准查找
    test_diameters = [8.1, 12.7, 26.5]
    for d in test_diameters:
        info = ThreadStandards.get_thread_standard_info(d)
        print(f"直径 {d}mm 最可能的标准: {info['thread_type']}, 标准直径: {info['standard_diameter']}mm, "
              f"螺距: {info['pitch']}mm, 角度: {info['angle']}°, 前缀: '{info['prefix']}'")