#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
螺纹分析工具模块：提供螺纹分析所需的各种工具函数
改进版本：增强特征识别准确性，减少误识别
"""

import logging
import math
from typing import Dict, Any, List, Tuple, Optional, Union

from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import (
    GeomAbs_Cylinder, GeomAbs_Circle, GeomAbs_BSplineCurve,
    GeomAbs_Line, GeomAbs_Plane
)
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.TopAbs import TopAbs_EDGE, TopAbs_FACE
from OCC.Core.TopoDS import topods_Edge, TopoDS_Face, TopoDS_Shape, topods_Face,topods
from OCC.Core.gp import gp_Dir, gp_Pnt, gp_Vec, gp_Cylinder, gp_Ax1, gp_Ax3
from OCC.Core.BRepGProp import brepgprop_LinearProperties, brepgprop_SurfaceProperties,brepgprop
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRep import BRep_Tool
from OCC.Core.ShapeAnalysis import ShapeAnalysis_Surface

from id2.thread_standards import ThreadStandards

# 尝试导入几何分析模块，如果不存在则提供基本实现
from id2.feature_analyzer_geometry import GeometryAnalyzer

logger = logging.getLogger(__name__)

class ThreadAnalysisUtils:
    """螺纹分析工具类，提供分析螺纹所需的各种工具函数"""
    
    # 螺纹识别的置信度阈值 - 增加到0.75以提高准确性
    THREAD_CONFIDENCE_THRESHOLD = 0.75
    
    # 螺纹特征判断的硬性条件数量 - 至少满足3个条件才可能是螺纹
    MIN_THREAD_FEATURES_REQUIRED = 3
    
    # 螺旋角度范围（度）- 典型螺纹的螺旋角度范围
    MIN_HELIX_ANGLE = 1.0
    MAX_HELIX_ANGLE = 45.0
    
    # 周期性检测的最小峰值强度比
    MIN_PERIODICITY_PEAK_RATIO = 2.5
    
    @staticmethod
    def analyze_edges(cylindrical_face: TopoDS_Face) -> Dict[str, Any]:
        """
        分析圆柱面的边缘特征，增强版本包含更多几何特性分析
        
        Args:
            cylindrical_face: 圆柱面
            
        Returns:
            dict: 边缘特征数据
        """
        result = {
            'edge_count': 0,
            'bspline_count': 0,
            'circle_count': 0,
            'line_count': 0,  # 新增：直线边的数量
            'edge_lengths': [],
            'edge_length_std': 0,
            'has_z_variation': False,
            'z_range': 0,
            'diameter': 0,
            'height_estimate': 0,
            'edge_density': 0,
            'has_circular_pattern': False,
            # 新增特征
            'helix_points': [],  # 螺旋路径采样点
            'z_period': 0.0,     # Z方向周期估计
            'z_period_confidence': 0.0,  # 周期估计的置信度
            'radial_variation': 0.0,  # 径向变化量
            'angles': [],        # 边缘角度采样
            'helix_angle': 0.0,  # 螺旋角度评估
            'is_conical': False, # 是否为锥形螺纹
            'topological_errors': 0,  # 拓扑错误计数
            'continuity_breaks': 0,   # 连续性中断计数
            'is_multi_thread': False, # 是否为多头螺纹
            'thread_count': 1,        # 螺纹头数
            'retreat_groove_detected': False,  # 是否检测到退刀槽
            'helical_continuity': 0.0, # 螺旋连续性评分
            'is_thread_like': False,   # 新增：是否具有螺纹特征
            'thread_confidence': 0.0,  # 新增：螺纹置信度评分
            'feature_scores': {},      # 新增：各项特征分数
            'non_thread_indicators': [], # 新增：非螺纹指示器
            'surface_area': 0.0,       # 新增：表面积
            'is_short_cylinder': False, # 新增：是否为短圆柱体
        }
        
        try:
            # 获取圆柱面基本信息
            surf = BRepAdaptor_Surface(cylindrical_face)
            if surf.GetType() != GeomAbs_Cylinder:
                # 非圆柱面，一定不是螺纹
                result['non_thread_indicators'].append("非圆柱面")
                result['is_thread_like'] = False
                result['thread_confidence'] = 0.0
                return result
                
            # 获取圆柱面信息
            cylinder = surf.Cylinder()
            result['diameter'] = cylinder.Radius() * 2
            cylinder_axis = cylinder.Axis()
            cylinder_location = cylinder.Location()
            # 获取圆柱体坐标系，用于后续分析
            cylinder_dir = cylinder_axis.Direction()
            
            # 计算表面积
            props = GProp_GProps()
            brepgprop.SurfaceProperties(cylindrical_face, props)
            result['surface_area'] = props.Mass()
            
            # 检查圆柱体高宽比 - 太短的可能不是螺纹
            # 这将在后面通过z_range计算
            
            # 用于检测螺旋连续性
            helix_segments = []
            # Z坐标分布，用于周期性检测
            z_distribution = {}
            # 半径采样，用于检测是否为锥形
            radius_samples = []
            # 角度采样，用于多头螺纹检测
            theta_samples = []
            # 存储螺旋角度样本
            helix_angle_samples = []
            
            # 检查边缘
            explorer = TopExp_Explorer(cylindrical_face, TopAbs_EDGE)
            edge_points = []
            z_coords = []
            edge_curves = []  # 存储曲线对象，用于拓扑分析
            
            # 记录边缘采样点的法线方向
            normal_vectors = []
            
            while explorer.More():
                edge = topods.Edge(explorer.Current())
                result['edge_count'] += 1
                
                # 计算边的长度
                props = GProp_GProps()
                try:
                    brepgprop.LinearProperties(edge, props)
                    length = props.Mass()
                    if length > 0:  # 确保长度有效
                        result['edge_lengths'].append(length)
                except Exception as e:
                    logger.warning(f"计算边长时出错: {str(e)}")
                
                # 分析边的类型
                try:
                    curve_adaptor = BRepAdaptor_Curve(edge)
                    curve_type = curve_adaptor.GetType()
                    edge_curves.append(curve_adaptor)
                    
                    if curve_type == GeomAbs_BSplineCurve:
                        result['bspline_count'] += 1
                        
                        # 增强版采样点收集，用于螺旋分析
                        first_param = curve_adaptor.FirstParameter()
                        last_param = curve_adaptor.LastParameter()
                        
                        # 参数范围有效性检查
                        if math.isfinite(first_param) and math.isfinite(last_param) and first_param < last_param:
                            # 收集更多采样点用于详细分析
                            sample_count = 15  # 增加采样点数量
                            helix_point_sequence = []
                            
                            for i in range(sample_count):
                                param = first_param + (last_param - first_param) * i / (sample_count - 1)
                                point = curve_adaptor.Value(param)
                                
                                # 计算点到轴的距离，检测锥度
                                dx = point.X() - cylinder_location.X()
                                dy = point.Y() - cylinder_location.Y()
                                r = math.sqrt(dx*dx + dy*dy)
                                radius_samples.append(r)
                                
                                # 计算角度用于多头螺纹检测
                                theta = math.atan2(dy, dx)
                                theta_samples.append(theta)
                                
                                # 记录采样点和z坐标
                                point_data = (point.X(), point.Y(), point.Z())
                                edge_points.append(point_data)
                                z_coords.append(point.Z())
                                
                                # 映射到圆柱坐标系，便于周期性分析
                                # 四舍五入到0.1mm，方便周期检测
                                z_rounded = round(point.Z() * 10) / 10
                                if z_rounded in z_distribution:
                                    z_distribution[z_rounded] += 1
                                else:
                                    z_distribution[z_rounded] = 1
                                
                                # 记录到螺旋点序列
                                helix_point_sequence.append((point.X(), point.Y(), point.Z()))
                                
                                # 获取曲线在此点的切线方向
                                if i > 0 and i < sample_count - 1:  # 排除端点
                                    try:
                                        point_gp = gp_Pnt()
                                        tangent = gp_Vec()
                                        curve_adaptor.D1(param, point_gp, tangent)
                                        
                                        # 计算在圆柱面上的法线方向
                                        center_dir = gp_Vec(
                                            point.X() - cylinder_location.X(),
                                            point.Y() - cylinder_location.Y(),
                                            0
                                        )
                                        if center_dir.Magnitude() > 1e-10:
                                            center_dir.Normalize()
                                            normal_vectors.append(center_dir)
                                        
                                        # 计算螺旋角度 - 切线与圆柱面轴向的夹角
                                        if tangent.Magnitude() > 1e-10:
                                            # 将切线投影到圆柱径向平面
                                            z_component = tangent.Z()
                                            xy_component = math.sqrt(tangent.X()**2 + tangent.Y()**2)
                                            
                                            # 计算螺旋角度（弧度）
                                            if xy_component > 1e-10:
                                                helix_angle = math.atan2(z_component, xy_component)
                                                helix_angle_deg = math.degrees(abs(helix_angle))
                                                # 只记录合理范围内的角度
                                                if 0.1 < helix_angle_deg < 89.9:
                                                    helix_angle_samples.append(helix_angle_deg)
                                    except Exception as e:
                                        logger.debug(f"计算切线失败: {e}")
                            
                            # 保存螺旋点序列用于连续性分析
                            if len(helix_point_sequence) > 2:
                                helix_segments.append(helix_point_sequence)
                                result['helix_points'].extend(helix_point_sequence)
                    
                    elif curve_type == GeomAbs_Circle:
                        result['circle_count'] += 1
                        
                        # 分析圆的位置和平面，用于退刀槽检测
                        try:
                            circle = curve_adaptor.Circle()
                            circle_radius = circle.Radius()
                            circle_location = circle.Location()
                            
                            # 如果圆的半径小于圆柱体半径的90%，可能是退刀槽
                            if circle_radius < cylinder.Radius() * 0.9:
                                result['retreat_groove_detected'] = True
                                
                            # 记录圆心坐标到z_coords
                            z_coords.append(circle_location.Z())
                        except Exception as e:
                            logger.warning(f"分析圆时出错: {str(e)}")
                    
                    elif curve_type == GeomAbs_Line:
                        # 记录直线边数量 - 直线过多可能暗示非螺纹
                        result['line_count'] += 1
                        
                except Exception as e:
                    logger.warning(f"分析边类型时出错: {str(e)}")
                
                explorer.Next()
            
            # 分析几何特性
            # 1. 计算高度估计（Z方向的范围）
            if z_coords:
                z_min, z_max = min(z_coords), max(z_coords)
                result['z_range'] = z_max - z_min
                result['height_estimate'] = result['z_range']
                result['has_z_variation'] = result['z_range'] > 0.5
                
                # 检查是否为短圆柱 - 短圆柱不太可能是螺纹
                if result['z_range'] < 0.5 * result['diameter']:
                    result['is_short_cylinder'] = True
                    result['non_thread_indicators'].append("短圆柱")
            
            # 2. 计算边长的标准差
            if result['edge_lengths']:
                mean_length = sum(result['edge_lengths']) / len(result['edge_lengths'])
                variance = sum((l - mean_length) ** 2 for l in result['edge_lengths']) / max(1, len(result['edge_lengths']))
                result['edge_length_std'] = math.sqrt(variance)
            
            # 3. 计算边缘密度（边数与高度的比值）
            if result['height_estimate'] > 0:
                result['edge_density'] = result['edge_count'] / result['height_estimate']
                
                # 螺纹通常会有合理的边缘密度
                if result['edge_density'] < 2.0:  # 太低的边缘密度不太可能是螺纹
                    result['non_thread_indicators'].append("边缘密度过低")
            
            # 4. 分析螺旋连续性
            if helix_segments:
                # 评估螺旋的连续性
                continuity_score = ThreadAnalysisUtils._evaluate_helix_continuity(helix_segments)
                result['helical_continuity'] = continuity_score
                
                # 检测连续性中断
                result['continuity_breaks'] = ThreadAnalysisUtils._detect_continuity_breaks(helix_segments)
                
                # 螺纹通常有良好的螺旋连续性
                if continuity_score < 0.6:
                    result['non_thread_indicators'].append("螺旋连续性差")
            else:
                # 没有检测到螺旋段，不太可能是螺纹
                result['non_thread_indicators'].append("无螺旋段")
            
            # 5. 使用FFT检测螺旋的周期性（螺距）
            if len(z_coords) > 10:
                try:
                    z_period, confidence = ThreadAnalysisUtils._detect_periodicity_fft(z_coords)
                    result['z_period'] = z_period
                    result['z_period_confidence'] = confidence
                    
                    # 周期性是螺纹的重要特征
                    if confidence < 0.6:
                        result['non_thread_indicators'].append("周期性不明显")
                    elif not (0.1 < z_period < 10.0):  # 非合理的螺距范围
                        result['non_thread_indicators'].append("非标准螺距")
                except Exception as e:
                    logger.warning(f"FFT周期检测失败: {e}")
                    result['non_thread_indicators'].append("周期检测失败")
            
            # 6. 分析半径样本以检测锥形螺纹
            if radius_samples:
                r_min, r_max = min(radius_samples), max(radius_samples)
                result['radial_variation'] = r_max - r_min
                
                # 如果半径变化超过10%，可能是锥形螺纹
                if result['radial_variation'] > 0.1 * result['diameter'] / 2:
                    result['is_conical'] = True
            
            # 7. 分析螺旋角度样本
            if helix_angle_samples:
                # 计算平均螺旋角度
                avg_helix_angle = sum(helix_angle_samples) / len(helix_angle_samples)
                result['helix_angle'] = avg_helix_angle
                
                # 检查螺旋角是否在合理范围内
                if not (ThreadAnalysisUtils.MIN_HELIX_ANGLE <= avg_helix_angle <= ThreadAnalysisUtils.MAX_HELIX_ANGLE):
                    result['non_thread_indicators'].append("非标准螺旋角")
            else:
                result['non_thread_indicators'].append("无螺旋角数据")
            
            # 8. 分析角度样本检测多头螺纹
            if theta_samples and len(theta_samples) > 10:
                try:
                    # 将角度排序并分析间隔
                    sorted_angles = sorted(theta_samples)
                    angle_diffs = []
                    for i in range(1, len(sorted_angles)):
                        diff = sorted_angles[i] - sorted_angles[i-1]
                        if diff > 0.1:  # 忽略太小的差异
                            angle_diffs.append(diff)
                    
                    if angle_diffs:
                        # 计算平均角度间隔
                        avg_angle_diff = sum(angle_diffs) / len(angle_diffs)
                        # 如果平均间隔小于π/2，可能是多头螺纹
                        if avg_angle_diff < math.pi / 2:
                            # 估计头数
                            est_threads = round(2 * math.pi / avg_angle_diff)
                            if est_threads > 1:
                                result['is_multi_thread'] = True
                                result['thread_count'] = est_threads
                except Exception as e:
                    logger.warning(f"多头螺纹检测失败: {e}")
            
            # 9. 检测是否存在环形模式
            # 增强的环形模式检测
            if result['circle_count'] >= 2:
                # 检查圆心的Z分布
                if z_coords:
                    # 计算相邻Z坐标的差值
                    sorted_z = sorted(z_coords)
                    z_diffs = [sorted_z[i] - sorted_z[i-1] for i in range(1, len(sorted_z))]
                    
                    if z_diffs:
                        # 计算Z差值的标准差
                        mean_z_diff = sum(z_diffs) / len(z_diffs)
                        z_diff_variance = sum((d - mean_z_diff) ** 2 for d in z_diffs) / len(z_diffs)
                        z_diff_std = math.sqrt(z_diff_variance)
                        
                        # 如果标准差小于平均值的30%，说明分布均匀，可能是环形模式
                        if z_diff_std < 0.3 * mean_z_diff and mean_z_diff > 0.1:
                            result['has_circular_pattern'] = True
            
            # 10. 拓扑一致性检查
            result['topological_errors'] = ThreadAnalysisUtils._check_topological_consistency(edge_curves)
            if result['topological_errors'] > 2:  # 拓扑错误过多，可能不是螺纹
                result['non_thread_indicators'].append("拓扑错误过多")
            
            # 11. 法线方向分析，用于内/外螺纹判断的补充信息
            if normal_vectors:
                # 如果法线大多指向外部，可能是外螺纹
                outward_count = 0
                for normal in normal_vectors:
                    # 计算法线与径向向量的点积
                    # 如果大于0，则指向外部
                    if normal.Dot(normal) > 0:
                        outward_count += 1
                
                if outward_count > len(normal_vectors) / 2:
                    result['normal_direction'] = 'outward'
                else:
                    result['normal_direction'] = 'inward'
            
            # 12. 圆柱面与相邻面关系分析 - 新增的判断
            # 寻找与圆柱面相邻的面，检查是否有可能是螺纹端面
            try:
                # 此处需要实现面关系分析，暂用简化判断
                # 完整实现需要使用TopoDS_Tool和TopExp_Explorer分析面间关系
                pass
            except Exception as e:
                logger.debug(f"面关系分析失败: {e}")
            
            # 13. B样条曲线与圆的比例 - 新增的判断
            # 螺纹通常B样条曲线比例较高
            if result['edge_count'] > 0:
                bspline_ratio = result['bspline_count'] / result['edge_count']
                # 如果B样条曲线比例过低，可能不是螺纹
                if bspline_ratio < 0.3:  # 低于30%的比例
                    result['non_thread_indicators'].append("B样条曲线比例过低")
            
            # 14. 直线比例过高检查 - 新增的判断
            if result['edge_count'] > 0:
                line_ratio = result['line_count'] / result['edge_count']
                # 如果直线比例过高，可能不是螺纹
                if line_ratio > 0.5:  # 超过50%的直线比例
                    result['non_thread_indicators'].append("直线比例过高")
            
            # 15. 对结果进行综合评估，判断是否为螺纹 - 新算法
            # 计算各项特征得分
            feature_scores = {
                '螺旋连续性': min(1.0, result['helical_continuity']),
                '周期性': min(1.0, result['z_period_confidence']) if 0.1 < result.get('z_period', 0) < 10.0 else 0.0,
                'B样条曲线比例': min(1.0, result['bspline_count'] / max(1, result['edge_count']) * 2.0),
                '边缘密度': min(1.0, result['edge_density'] / 10.0) if result['edge_density'] > 2.0 else 0.0,
                '螺旋角度': 1.0 if ThreadAnalysisUtils.MIN_HELIX_ANGLE <= result.get('helix_angle', 0) <= ThreadAnalysisUtils.MAX_HELIX_ANGLE else 0.0,
                'Z方向变化': 1.0 if result['has_z_variation'] else 0.0,
                '非短圆柱': 0.0 if result['is_short_cylinder'] else 1.0,
                '拓扑一致性': max(0.0, 1.0 - result['topological_errors'] / 5.0),
            }
            
            result['feature_scores'] = feature_scores
            
            # 计算综合置信度分数
            weights = {
                '螺旋连续性': 0.25,
                '周期性': 0.25,
                'B样条曲线比例': 0.15,
                '边缘密度': 0.10,
                '螺旋角度': 0.15,
                'Z方向变化': 0.05,
                '非短圆柱': 0.05,
                '拓扑一致性': 0.05,
            }
            
            confidence = sum(score * weights[key] for key, score in feature_scores.items())
            result['thread_confidence'] = confidence
            
            # 检查硬性条件数量
            hard_features_met = sum(1 for score in feature_scores.values() if score > 0.7)
            
            # 最终判断是否为螺纹
            result['is_thread_like'] = (
                confidence >= ThreadAnalysisUtils.THREAD_CONFIDENCE_THRESHOLD and
                hard_features_met >= ThreadAnalysisUtils.MIN_THREAD_FEATURES_REQUIRED and
                len(result['non_thread_indicators']) <= 2  # 不超过2个否定指标
            )
            
        except Exception as e:
            logger.error(f"分析边缘时发生错误: {str(e)}")
            result['non_thread_indicators'].append(f"分析错误: {str(e)}")
            result['is_thread_like'] = False
        
        return result
    
    @staticmethod
    def _detect_periodicity_fft(values: List[float]) -> Tuple[float, float]:
        """
        使用FFT检测数列的周期性，增强版返回周期和置信度
        
        Args:
            values: 输入数值列表
            
        Returns:
            Tuple[float, float]: 检测到的周期和置信度(0-1)
        """
        try:
            # 尝试导入numpy
            import numpy as np
            from scipy import signal
            
            # 对数值进行预处理，确保等间隔
            # 首先排序
            z_sorted = np.sort(np.array(values))
            
            # 计算频谱
            fft_result = np.abs(np.fft.fft(z_sorted - np.mean(z_sorted)))
            
            # 只保留前半部分（因为FFT结果是对称的）
            n = len(fft_result) // 2
            fft_result = fft_result[1:n]  # 忽略直流分量
            
            if len(fft_result) < 3:
                return 0, 0.0
                
            # 找到最强的频率
            peak_idx = np.argmax(fft_result)
            if peak_idx == 0:
                # 没有找到明显的周期
                return 0, 0.0
            
            peak_value = fft_result[peak_idx]
            
            # 计算峰值与平均值的比值，作为置信度
            avg_value = np.mean(fft_result)
            if avg_value < 1e-10:
                confidence = 0.0
            else:
                # 峰值比越高，置信度越高
                ratio = peak_value / avg_value
                confidence = min(1.0, ratio / ThreadAnalysisUtils.MIN_PERIODICITY_PEAK_RATIO)
                
            # 计算周期
            period = len(values) / (peak_idx + 1)
            
            # 换算为实际距离
            z_range = max(values) - min(values)
            actual_period = z_range / period
            
            # 合理性检查
            if 0.1 <= actual_period <= 10.0:  # 合理的螺距范围
                return actual_period, confidence
            else:
                return actual_period, 0.0  # 周期不在合理范围内，置信度为0
            
        except ImportError:
            # 如果没有numpy，使用简化方法检测周期
            logger.info("numpy/scipy未安装，使用简化周期检测算法")
            period = ThreadAnalysisUtils._simple_periodicity_detection(values)
            # 简化版无法提供置信度，给一个保守的估计
            confidence = 0.5 if 0.1 <= period <= 10.0 else 0.0
            return period, confidence
    
    @staticmethod
    def _simple_periodicity_detection(values: List[float]) -> float:
        """
        简化版的周期性检测，基于自相关
        
        Args:
            values: 输入数值列表
            
        Returns:
            float: 检测到的周期
        """
        if len(values) < 10:
            return 0
            
        # 排序数值
        sorted_values = sorted(values)
        
        # 计算差值
        diffs = [sorted_values[i] - sorted_values[i-1] for i in range(1, len(sorted_values))]
        
        # 过滤掉太小的差值
        filtered_diffs = [d for d in diffs if d > 0.01]
        
        if not filtered_diffs:
            return 0
            
        # 计算平均差值作为周期估计
        avg_diff = sum(filtered_diffs) / len(filtered_diffs)
        
        # 检查差值的一致性
        std_dev = math.sqrt(sum((d - avg_diff)**2 for d in filtered_diffs) / len(filtered_diffs))
        
        # 如果标准差太大，周期性不明显
        if std_dev > 0.5 * avg_diff:
            return 0
            
        return avg_diff
    
    @staticmethod
    def _evaluate_helix_continuity(helix_segments: List[List[Tuple[float, float, float]]]) -> float:
        """
        评估螺旋线的连续性
        
        Args:
            helix_segments: 螺旋线段列表
            
        Returns:
            float: 连续性评分，0-1之间，1表示完全连续
        """
        if not helix_segments or len(helix_segments) < 2:
            return 0.0
            
        # 连续性评分
        continuity_score = 1.0
        
        # 检查每个螺旋段之间的连接
        for i in range(len(helix_segments) - 1):
            segment1 = helix_segments[i]
            segment2 = helix_segments[i + 1]
            
            if not segment1 or not segment2:
                continuity_score *= 0.5
                continue
                
            # 计算段之间的断开距离
            end_point = segment1[-1]
            start_point = segment2[0]
            
            distance = math.sqrt(
                (end_point[0] - start_point[0])**2 +
                (end_point[1] - start_point[1])**2 +
                (end_point[2] - start_point[2])**2
            )
            
            # 如果距离太大，说明连续性差
            if distance > 1.0:  # 1mm作为阈值
                continuity_score *= 0.7
                
            # 检查方向连续性
            if len(segment1) > 2 and len(segment2) > 2:
                # 计算结束段的方向
                dir1 = (
                    segment1[-1][0] - segment1[-2][0],
                    segment1[-1][1] - segment1[-2][1],
                    segment1[-1][2] - segment1[-2][2]
                )
                
                # 计算开始段的方向
                dir2 = (
                    segment2[1][0] - segment2[0][0],
                    segment2[1][1] - segment2[0][1],
                    segment2[1][2] - segment2[0][2]
                )
                
                # 计算方向的夹角余弦值
                dir1_mag = math.sqrt(dir1[0]**2 + dir1[1]**2 + dir1[2]**2)
                dir2_mag = math.sqrt(dir2[0]**2 + dir2[1]**2 + dir2[2]**2)
                
                if dir1_mag > 1e-6 and dir2_mag > 1e-6:
                    dot_product = dir1[0]*dir2[0] + dir1[1]*dir2[1] + dir1[2]*dir2[2]
                    cos_angle = dot_product / (dir1_mag * dir2_mag)
                    
                    # 如果夹角很大，连续性差
                    if cos_angle < 0.7:  # 约45°以上的角度
                        continuity_score *= 0.8
        
        return max(0.0, min(1.0, continuity_score))
    
    @staticmethod
    def _detect_continuity_breaks(helix_segments: List[List[Tuple[float, float, float]]]) -> int:
        """
        检测螺旋线中的连续性中断
        
        Args:
            helix_segments: 螺旋线段列表
            
        Returns:
            int: 连续性中断的数量
        """
        if not helix_segments:
            return 0
            
        break_count = 0
        
        # 1. 检查段间断裂
        for i in range(len(helix_segments) - 1):
            segment1 = helix_segments[i]
            segment2 = helix_segments[i + 1]
            
            if not segment1 or not segment2:
                break_count += 1
                continue
                
            # 计算段之间的断开距离
            end_point = segment1[-1]
            start_point = segment2[0]
            
            distance = math.sqrt(
                (end_point[0] - start_point[0])**2 +
                (end_point[1] - start_point[1])**2 +
                (end_point[2] - start_point[2])**2
            )
            
            # 如果距离太大，认为是一次中断
            if distance > 1.0:  # 1mm作为阈值
                break_count += 1
        
        # 2. 检查段内急剧变化
        for segment in helix_segments:
            if len(segment) < 3:
                continue
                
            for i in range(1, len(segment) - 1):
                p1 = segment[i-1]
                p2 = segment[i]
                p3 = segment[i+1]
                
                # 计算两段的方向向量
                v1 = (p2[0] - p1[0], p2[1] - p1[1], p2[2] - p1[2])
                v2 = (p3[0] - p2[0], p3[1] - p2[1], p3[2] - p2[2])
                
                # 计算向量的大小
                v1_mag = math.sqrt(v1[0]**2 + v1[1]**2 + v1[2]**2)
                v2_mag = math.sqrt(v2[0]**2 + v2[1]**2 + v2[2]**2)
                
                if v1_mag > 1e-6 and v2_mag > 1e-6:
                    # 计算夹角余弦值
                    dot_product = v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2]
                    cos_angle = dot_product / (v1_mag * v2_mag)
                    
                    # 如果夹角很大，认为是连续性中断
                    if cos_angle < 0.5:  # 约60°以上的角度
                        break_count += 1
        
        return break_count
    
    @staticmethod
    def _check_topological_consistency(edge_curves: List) -> int:
        """
        检查螺纹面的拓扑一致性
        
        Args:
            edge_curves: 边缘曲线列表
            
        Returns:
            int: 发现的拓扑问题数量
        """
        if not edge_curves or len(edge_curves) < 2:
            return 0
            
        issues_count = 0
        
        # 检查曲线的曲率变化
        for curve in edge_curves:
            try:
                # 采样点检查曲率变化
                params = []
                first_param = curve.FirstParameter()
                last_param = curve.LastParameter()
                
                if math.isfinite(first_param) and math.isfinite(last_param) and first_param < last_param:
                    for i in range(5):
                        params.append(first_param + (last_param - first_param) * i / 4)
                
                    # 计算曲率变化
                    curvatures = []
                    for param in params:
                        point = gp_Pnt()
                        d1 = gp_Vec()
                        d2 = gp_Vec()
                        
                        try:
                            curve.D2(param, point, d1, d2)
                            
                            # 计算曲率
                            if d1.Magnitude() > 1e-10:
                                cross_product = d1.Crossed(d2)
                                curvature = cross_product.Magnitude() / (d1.Magnitude() ** 3)
                                curvatures.append(curvature)
                        except Exception:
                            # 如果计算失败，可能是拓扑问题
                            issues_count += 1
                            continue
                    
                    # 检查曲率的突变
                    if len(curvatures) > 2:
                        for i in range(1, len(curvatures)):
                            if curvatures[i-1] > 1e-10 and curvatures[i] > 1e-10:
                                ratio = max(curvatures[i], curvatures[i-1]) / min(curvatures[i], curvatures[i-1])
                                if ratio > 5.0:  # 曲率突变超过5倍
                                    issues_count += 1
            except Exception:
                issues_count += 1
        
        return issues_count
    
    @staticmethod
    def estimate_thread_pitch(edge_data: Dict[str, Any]) -> float:
        """
        估计螺纹螺距
        
        Args:
            edge_data: 边缘分析数据
            
        Returns:
            float: 估计的螺距
        """
        # 检查是否为螺纹
        if not edge_data.get('is_thread_like', False):
            return 0.0
            
        # 首先尝试使用FFT分析结果
        if 'z_period' in edge_data and edge_data['z_period'] > 0:
            return edge_data['z_period']
        
        # 或者使用传统方法估计
        if edge_data['z_range'] > 0 and edge_data['bspline_count'] > 0:
            # 估计每圈的高度变化
            pitch_estimate = edge_data['z_range'] / max(1, edge_data['bspline_count']) * 2
            
            # 检查是否在合理范围内
            if 0.1 < pitch_estimate < 10.0:
                return pitch_estimate
        
        return 0.0
    
    @staticmethod
    def determine_thread_type(cylinder: gp_Cylinder, axis: gp_Ax1, radius: float, 
                             edge_data: Dict[str, Any], shape: TopoDS_Shape) -> Dict[str, Any]:
        """
        判断内螺纹还是外螺纹
        
        Args:
            cylinder: 圆柱体
            axis: 圆柱轴
            radius: 半径
            edge_data: 边缘分析数据
            shape: 整体形状
            
        Returns:
            Dict[str, Any]: 螺纹类型信息
        """
        result = {}
        
        # 首先检查是否为螺纹
        if not edge_data.get('is_thread_like', False):
            result['类型'] = '非螺纹特征'
            result['置信度'] = 0.0
            return result
            
        try:
            # 方法1: 使用法线方向
            inward_hint = False
            if 'normal_direction' in edge_data:
                inward_hint = edge_data['normal_direction'] == 'inward'
            
            # 方法2: 使用几何点检测
            # 获取圆柱面的轴向方向向量
            direction = axis.Direction()
            axis_dir = gp_Vec(direction.X(), direction.Y(), direction.Z())
            
            # 检查是否是零向量
            mag = axis_dir.Magnitude()
            if mag < 1e-10:
                # 默认假设
                result['类型'] = '内螺纹' if inward_hint else '外螺纹'
                result['置信度'] = 0.6  # 中等置信度
            else:
                # 规范化方向向量
                axis_dir.Multiply(1.0/mag)
                
                # 创建圆柱内部的检测点 - 使用多个点提高准确性
                center_point = gp_Pnt(
                    cylinder.Location().X(), 
                    cylinder.Location().Y(), 
                    cylinder.Location().Z()
                )
                
                # 使用多个角度的检测点
                sampling_points = []
                sampling_results = []
                
                # 创建更多检测点以提高准确性
                for angle in [0, math.pi/4, math.pi/2, 3*math.pi/4, math.pi, 5*math.pi/4, 3*math.pi/2, 7*math.pi/4]:
                    # 创建一个与轴向垂直的向量
                    normal_dir = gp_Vec(math.cos(angle), math.sin(angle), 0)
                    # 确保向量与轴向垂直
                    normal_dir = normal_dir.Crossed(axis_dir).Crossed(axis_dir)
                    if normal_dir.Magnitude() > 1e-10:
                        normal_dir.Normalize()
                        
                        # 计算向内部偏移的点 - 使用更小的偏移以提高准确性
                        offset = radius * 0.3  # 减小偏移距离
                        test_point = gp_Pnt(
                            center_point.X() + normal_dir.X() * offset,
                            center_point.Y() + normal_dir.Y() * offset,
                            center_point.Z() + normal_dir.Z() * offset
                        )
                        
                        # 检查点是否在形状内部
                        is_inside = GeometryAnalyzer.check_point_in_shape(test_point, shape)
                        sampling_points.append((test_point, is_inside))
                        sampling_results.append(is_inside)
                
                # 根据多数投票确定内/外螺纹
                inside_votes = sum(1 for r in sampling_results if r)
                outside_votes = len(sampling_results) - inside_votes
                
                # 考虑法线信息
                if inward_hint:
                    inside_votes += 2  # 增加法线方向的权重
                else:
                    outside_votes += 2
                
                # 记录调试信息
                result['内部点数'] = inside_votes
                result['外部点数'] = outside_votes
                
                # 计算置信度
                total_votes = inside_votes + outside_votes
                if total_votes > 0:
                    if inside_votes > outside_votes:
                        confidence = inside_votes / total_votes
                        result['类型'] = '内螺纹'
                    else:
                        confidence = outside_votes / total_votes
                        result['类型'] = '外螺纹'
                    result['置信度'] = confidence
                else:
                    result['类型'] = '未知螺纹类型'
                    result['置信度'] = 0.5
        except Exception as e:
            logger.error(f"判断螺纹类型时出错: {str(e)}")
            result['类型'] = '未知螺纹类型'
            result['类型错误'] = str(e)
            result['置信度'] = 0.0
        
        return result
    
    @staticmethod
    def estimate_thread_angle(edge_data: Dict[str, Any], thread_type: str) -> float:
        """
        估计螺纹角度
        
        Args:
            edge_data: 边缘分析数据
            thread_type: 螺纹类型（公制/英制/管螺纹）
            
        Returns:
            float: 估计的螺纹角度（度）
        """
        # 检查是否为螺纹
        if not edge_data.get('is_thread_like', False):
            return 0.0
            
        # 根据边缘角度样本估计
        if 'angles' in edge_data and edge_data['angles']:
            # 使用边缘角度的统计
            angles = edge_data['angles']
            if len(angles) > 3:  # 至少需要几个样本
                avg_angle = sum(angles) / len(angles)
                return avg_angle
        
        # 或者根据螺纹类型确定标准角度
        if thread_type == "公制":
            return ThreadStandards.THREAD_ANGLES['M']
        elif thread_type == "英制":
            return ThreadStandards.THREAD_ANGLES['UNC']
        elif thread_type == "管螺纹" and edge_data.get('is_conical', False):
            return ThreadStandards.THREAD_ANGLES['NPT']
        elif thread_type == "管螺纹":
            return ThreadStandards.THREAD_ANGLES['G']
        
        # 默认返回公制标准角度
        return 60.0
    
    @staticmethod
    def format_thread_specification(thread_type: str, properties: Dict[str, Any], 
                                   diameter: float) -> str:
        """
        格式化螺纹规格
        
        Args:
            thread_type: 螺纹类型
            properties: 螺纹属性
            diameter: 螺纹直径
            
        Returns:
            str: 格式化的螺纹规格
        """
        # 检查是否为螺纹
        if thread_type == "非螺纹特征":
            return "非螺纹特征"
            
        if thread_type == "公制":
            if '标准螺距' in properties:
                return f"M{properties['标准直径']}x{properties['标准螺距']}"
            else:
                return f"M{properties['标准直径']}"
                
        elif thread_type == "英制":
            # 将直径转换为分数英寸表示
            inch_diameter = diameter / 25.4
            
            # 查找映射到标准分数
            if diameter in ThreadStandards.INCH_SIZE_MAP:
                inch_str = ThreadStandards.INCH_SIZE_MAP[diameter]
            else:
                # 简化为常用分数表示
                if abs(inch_diameter - 1/4) < 0.01:
                    inch_str = "1/4"
                elif abs(inch_diameter - 5/16) < 0.01:
                    inch_str = "5/16"
                elif abs(inch_diameter - 3/8) < 0.01:
                    inch_str = "3/8"
                elif abs(inch_diameter - 1/2) < 0.01:
                    inch_str = "1/2"
                elif abs(inch_diameter - 5/8) < 0.01:
                    inch_str = "5/8"
                elif abs(inch_diameter - 3/4) < 0.01:
                    inch_str = "3/4"
                elif abs(inch_diameter - 1) < 0.01:
                    inch_str = "1"
                else:
                    inch_str = f"{inch_diameter:.3f}"
            
            # 计算TPI (Threads Per Inch)
            if '标准螺距' in properties and properties['标准螺距'] > 0:
                tpi = round(25.4 / properties['标准螺距'])
                return f"{inch_str}\"-{tpi}"
            else:
                return f"{inch_str}\""
                
        elif thread_type == "管螺纹":
            # 查找管螺纹尺寸
            if diameter in ThreadStandards.PIPE_SIZE_MAP:
                pipe_size = ThreadStandards.PIPE_SIZE_MAP[diameter]
            else:
                # 按照接近程度寻找
                closest_diameter = min(ThreadStandards.PIPE_SIZE_MAP.keys(), key=lambda x: abs(x - diameter))
                pipe_size = ThreadStandards.PIPE_SIZE_MAP[closest_diameter]
            
            # 根据是否为锥形确定前缀
            if properties.get('锥形螺纹', False):
                return f"NPT {pipe_size}"
            else:
                return f"G {pipe_size}"
        
        # 未知类型
        return f"Unknown {diameter}mm"
    
    @staticmethod
    def analyze_cylindrical_face(face: TopoDS_Face, shape: TopoDS_Shape) -> Dict[str, Any]:
        """
        分析圆柱面是否为螺纹面，集成所有判断逻辑
        
        Args:
            face: 要分析的圆柱面
            shape: 整体形状
            
        Returns:
            Dict[str, Any]: 分析结果
        """
        result = {
            'is_thread': False,
            'thread_type': '非螺纹特征',
            'confidence': 0.0,
            'details': {}
        }
        
        try:
            # 1. 检查是否为圆柱面
            surf = BRepAdaptor_Surface(face)
            if surf.GetType() != GeomAbs_Cylinder:
                return result
                
            # 2. 获取圆柱面信息
            cylinder = surf.Cylinder()
            radius = cylinder.Radius()
            axis = cylinder.Axis()
            
            # 3. 分析边缘特征
            edge_data = ThreadAnalysisUtils.analyze_edges(face)
            
            # 4. 判断是否为螺纹
            result['is_thread'] = edge_data.get('is_thread_like', False)
            result['confidence'] = edge_data.get('thread_confidence', 0.0)
            result['details'] = edge_data
            
            # 如果不是螺纹，直接返回
            if not result['is_thread']:
                return result
                
            # 5. 确定螺纹类型（内/外）
            thread_type_info = ThreadAnalysisUtils.determine_thread_type(
                cylinder, axis, radius, edge_data, shape
            )
            
            result['thread_type'] = thread_type_info.get('类型', '未知螺纹类型')
            result['thread_direction_confidence'] = thread_type_info.get('置信度', 0.5)
            
            # 6. 估计螺距
            pitch = ThreadAnalysisUtils.estimate_thread_pitch(edge_data)
            result['pitch'] = pitch
            
            # 7. 确定螺纹标准(公制/英制/管螺纹)
            # 这里可以根据直径和螺距进行标准判断
            standard = "公制"  # 默认为公制
            
            # 8. 生成螺纹规格
            properties = {
                '标准直径': round(2 * radius * 10) / 10,  # 四舍五入到0.1mm
                '标准螺距': round(pitch * 10) / 10 if pitch > 0 else None
            }
            
            result['thread_specification'] = ThreadAnalysisUtils.format_thread_specification(
                standard, properties, 2 * radius
            )
            
            # 9. 估计螺纹角度
            thread_angle = ThreadAnalysisUtils.estimate_thread_angle(edge_data, standard)
            result['thread_angle'] = thread_angle
            
        except Exception as e:
            logger.error(f"分析圆柱面时发生错误: {str(e)}")
            result['error'] = str(e)
        
        return result


if __name__ == "__main__":
    # 测试代码
    logging.basicConfig(level=logging.INFO)
    logger.info("改进的螺纹分析工具模块 - 提供更准确的螺纹分析功能")