#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
特征拓扑关系分析模块：识别和分析特征之间的拓扑关系
包括相邻、包含、相交、同轴、平行、垂直等关系
"""

import logging
import math
import numpy as np
from typing import List, Dict, Any, Tuple, Set
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.BRep import BRep_Tool
from OCC.Core.BRepExtrema import BRepExtrema_DistShapeShape
from OCC.Core.BRepAlgoAPI import BRepAlgoAPI_Common, BRepAlgoAPI_Section
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.TopAbs import TopAbs_EDGE, TopAbs_VERTEX, TopAbs_FACE
from OCC.Core.TopoDS import topods
from OCC.Core.gp import gp_Pnt, gp_Vec, gp_Dir, gp_Ax1
from OCC.Core.GeomAbs import GeomAbs_Plane, GeomAbs_Cylinder
from OCC.Core.TopTools import TopTools_MapOfShape
from OCC.Extend.TopologyUtils import TopologyExplorer
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop

logger = logging.getLogger(__name__)

class TopologyRelation:
    """拓扑关系类型枚举"""
    ADJACENT = "相邻"  # 共享边或顶点 - 影响加工
    COAXIAL = "同轴"  # 同一轴线 - 影响加工
    PARALLEL = "平行"  # 平行关系 - 影响加工
    PERPENDICULAR = "垂直"  # 垂直关系 - 影响加工
    CONCENTRIC = "同心"  # 同心圆/圆柱 - 影响加工
    TANGENT = "相切"  # 相切关系 - 影响加工
    INTERSECTING = "相交"  # 相交但不相邻 - 影响加工
    CONTAINING = "包含"  # 一个特征包含另一个 - 影响加工
    CONTAINED = "被包含"  # 被另一个特征包含 - 影响加工
    COPLANAR = "共面"  # 在同一平面上 - 影响加工
    OFFSET = "偏移"  # 偏移关系（平行且有固定距离）
    SYMMETRIC = "对称"  # 对称关系
    
    # 定义影响加工的关系类型集合
    MANUFACTURING_RELEVANT = {
        ADJACENT,      # 相邻面需要考虑刀具干涉
        COAXIAL,       # 同轴孔可能需要同一次装夹
        PARALLEL,      # 平行面影响装夹和加工顺序
        PERPENDICULAR, # 垂直面影响刀具选择和路径
        CONCENTRIC,    # 同心特征可能共用加工基准
        TANGENT,       # 相切面需要特殊的刀具路径
        INTERSECTING,  # 相交特征影响加工顺序
        CONTAINING,    # 包含关系影响加工顺序
        CONTAINED,     # 被包含特征需要先加工
        COPLANAR      # 共面特征可能一次加工
    }

class  FeatureTopologyAnalyzer:
    """特征拓扑关系分析器"""
    
    def __init__(self, features: List[Any], tolerance: float = 1e-6, manufacturing_only: bool = True):
        """
        初始化拓扑关系分析器
        
        Args:
            features: 机械特征对象列表
            tolerance: 几何容差
            manufacturing_only: 是否只分析影响加工的关系
        """
        self.features = features
        self.tolerance = tolerance
        self.manufacturing_only = manufacturing_only
        self.topology_relations = []
        
        # 建立特征索引映射
        self.feature_index_map = {id(f): i for i, f in enumerate(features)}
        
        # 缓存特征的几何信息
        self.feature_geometry_cache = {}
        self._cache_feature_geometry()
    
    def _cache_feature_geometry(self):
        """缓存所有特征的几何信息以提高性能"""
        for i, feature in enumerate(self.features):
            geometry_info = self._extract_geometry_info(feature)
            self.feature_geometry_cache[i] = geometry_info
    
    def _extract_geometry_info(self, feature) -> Dict[str, Any]:
        """
        提取特征的几何信息
        
        Args:
            feature: 机械特征对象
            
        Returns:
            dict: 几何信息字典
        """
        info = {
            'type': feature.特征类型,
            'shape': feature.形状,
            'properties': feature.属性.copy()
        }
        
        try:
            # 提取表面信息
            surf = BRepAdaptor_Surface(feature.形状)
            surf_type = surf.GetType()
            
            if surf_type == GeomAbs_Plane:
                plane = surf.Plane()
                info['surface_type'] = 'plane'
                info['location'] = self._gp_pnt_to_tuple(plane.Location())
                info['normal'] = self._gp_dir_to_tuple(plane.Axis().Direction())
                
            elif surf_type == GeomAbs_Cylinder:
                cylinder = surf.Cylinder()
                info['surface_type'] = 'cylinder'
                info['location'] = self._gp_pnt_to_tuple(cylinder.Location())
                info['axis'] = self._gp_dir_to_tuple(cylinder.Axis().Direction())
                info['radius'] = cylinder.Radius()
                
            # 提取边界框信息
            info['edges'] = self._extract_edges(feature.形状)
            info['vertices'] = self._extract_vertices(feature.形状)
            
        except Exception as e:
            logger.warning(f"提取特征几何信息时出错: {str(e)}")
        
        return info
    
    def _gp_pnt_to_tuple(self, point: gp_Pnt) -> Tuple[float, float, float]:
        """将gp_Pnt转换为元组"""
        return (point.X(), point.Y(), point.Z())
    
    def _gp_dir_to_tuple(self, direction: gp_Dir) -> Tuple[float, float, float]:
        """将gp_Dir转换为元组"""
        return (direction.X(), direction.Y(), direction.Z())
    
    def _extract_edges(self, shape) -> List[Any]:
        """提取形状的边"""
        edges = []
        explorer = TopExp_Explorer(shape, TopAbs_EDGE)
        while explorer.More():
            edge = topods.Edge(explorer.Current())
            edges.append(edge)
            explorer.Next()
        return edges
    
    def _extract_vertices(self, shape) -> List[Tuple[float, float, float]]:
        """提取形状的顶点坐标"""
        vertices = []
        explorer = TopExp_Explorer(shape, TopAbs_VERTEX)
        while explorer.More():
            vertex = topods.Vertex(explorer.Current())
            pnt = BRep_Tool.Pnt(vertex)
            vertices.append(self._gp_pnt_to_tuple(pnt))
            explorer.Next()
        return vertices
    
    def analyze_all_relations(self) -> List[Dict[str, Any]]:
        """
        分析所有特征之间的拓扑关系
        
        Returns:
            list: 拓扑关系列表（合并了相同特征对的多种关系）
        """
        raw_relations = []
        n = len(self.features)
        
        # 遍历所有特征对
        for i in range(n):
            for j in range(i + 1, n):
                feature_relations = self._analyze_feature_pair(i, j)
                if feature_relations:
                    raw_relations.extend(feature_relations)
        
        # 合并相同特征对的多种关系
        merged_relations = self._merge_relations(raw_relations)
        
        self.topology_relations = merged_relations
        return merged_relations
    
    def _merge_relations(self, relations: List[Dict[str, Any]]) -> List[Dict[str, Any]]:
        """
        合并相同特征对之间的多种拓扑关系,并过滤出影响加工的关系
        
        优化点:
        1. 统一关系类型字段,避免重复
        2. 扁平化 details 结构
        3. 减少不必要的中间数据
        """
        # 如果只需要影响加工的关系,先过滤
        if self.manufacturing_only:
            relations = [
                rel for rel in relations 
                if rel['relation_type'] in TopologyRelation.MANUFACTURING_RELEVANT
            ]
        
        # 使用字典来组织关系,key是特征对的标识
        relation_map = {}
        
        for relation in relations:
            # 创建特征对的唯一标识(保证较小的索引在前)
            idx1, idx2 = relation['feature1_index'], relation['feature2_index']
            if idx1 > idx2:
                idx1, idx2 = idx2, idx1
            
            pair_key = (idx1, idx2)
            
            if pair_key not in relation_map:
                # 第一次遇到这个特征对,创建新的合并关系
                relation_map[pair_key] = {
                    'feature1_index': idx1,
                    'feature2_index': idx2,
                    'feature1_type': self.features[idx1].特征类型,
                    'feature2_type': self.features[idx2].特征类型,
                    'relation_types': [],  # 存储多个关系类型
                    'details': {}  # 扁平化存储所有细节
                }
            
            # 添加关系类型(避免重复)
            rel_type = relation['relation_type']
            if rel_type not in relation_map[pair_key]['relation_types']:
                relation_map[pair_key]['relation_types'].append(rel_type)
            
            # 合并细节信息(扁平化,避免嵌套)
            if relation.get('details'):
                for key, value in relation['details'].items():
                    # 如果key已存在且值不同,使用列表存储多个值
                    if key in relation_map[pair_key]['details']:
                        existing = relation_map[pair_key]['details'][key]
                        if existing != value:
                            if not isinstance(existing, list):
                                relation_map[pair_key]['details'][key] = [existing]
                            if value not in relation_map[pair_key]['details'][key]:
                                relation_map[pair_key]['details'][key].append(value)
                    else:
                        relation_map[pair_key]['details'][key] = value
        
        # 转换回列表格式,优化输出结构
        merged_relations = []
        for pair_key, merged_relation in relation_map.items():
            relation_types = merged_relation['relation_types']
            
            # 优化:统一使用 relation_types 字段
            if len(relation_types) == 1:
                merged_relation['relation_type'] = relation_types[0]
                merged_relation['is_multi'] = False
            else:
                merged_relation['relation_type'] = ', '.join(relation_types)
                merged_relation['is_multi'] = True
            
            # 移除临时的 relation_types 列表,避免冗余
            # 如果需要完整列表,可以从 relation_type 字符串解析
            del merged_relation['relation_types']
            
            merged_relations.append(merged_relation)
        
        return merged_relations
    
    
    def _create_relation(self, idx1: int, idx2: int, 
                        relation_type: str, 
                        details: Dict[str, Any]) -> Dict[str, Any]:
        """
        创建关系记录 - 优化版
        
        优化点:
        1. 最小化必要字段
        2. 避免存储可计算的数据
        """
        return {
            'feature1_index': idx1,
            'feature2_index': idx2,
            'relation_type': relation_type,
            'details': details  # 只存储关键信息
            # 移除 feature_type,可以从 features 列表获取
        }
    
    
    def build_topology_graph(self) -> Dict[str, Any]:
        """
        构建拓扑关系图 - 优化版
        
        优化点:
        1. 减少节点中的冗余属性
        2. 简化边的结构
        """
        graph = {
            'nodes': [],
            'edges': []
        }
        
        # 添加节点(特征) - 最小化数据
        for i, feature in enumerate(self.features):
            node = {
                'id': i,
                'type': feature.特征类型
                # 移除 properties,需要时可从原始 features 获取
            }
            graph['nodes'].append(node)
        
        # 添加边(关系) - 简化结构
        for relation in self.topology_relations:
            edge = {
                'source': relation['feature1_index'],
                'target': relation['feature2_index'],
                'type': relation['relation_type'],
                'multi': relation.get('is_multi', False)
                # details 可选,需要时再添加
            }
            
            # 只有当 details 包含关键信息时才添加
            if relation.get('details'):
                # 过滤掉冗余或不重要的 details
                important_details = self._filter_important_details(relation['details'])
                if important_details:
                    edge['details'] = important_details
            
            graph['edges'].append(edge)
        
        return graph
    
    
    def _filter_important_details(self, details: Dict[str, Any]) -> Dict[str, Any]:
        """
        过滤出重要的细节信息,减少数据冗余
        
        Args:
            details: 原始细节字典
            
        Returns:
            过滤后的重要细节
        """
        # 定义重要字段的白名单
        important_keys = {
            'distance', 'angle', 'axis', 'offset_distance',
            'shared_edges', 'spacing', 'radius'
        }
        
        filtered = {}
        for key, value in details.items():
            # 只保留白名单中的字段
            if key in important_keys:
                # 如果是数值,四舍五入减少精度冗余
                if isinstance(value, float):
                    filtered[key] = round(value, 3)
                else:
                    filtered[key] = value
        
        return filtered
    
    
    def get_feature_connections(self, feature_index: int) -> List[Dict[str, Any]]:
        """
        获取指定特征的所有连接关系 - 优化版
        
        优化点:
        1. 减少返回数据的冗余
        2. 按需提供详细信息
        """
        connections = []
        for relation in self.topology_relations:
            if relation['feature1_index'] == feature_index or \
               relation['feature2_index'] == feature_index:
                
                # 确定连接的另一端
                connected_idx = (relation['feature2_index'] 
                               if relation['feature1_index'] == feature_index 
                               else relation['feature1_index'])
                
                # 简化的连接信息
                connection = {
                    'connected_to': connected_idx,
                    'relation': relation['relation_type'],
                    'multi': relation.get('is_multi', False)
                }
                
                # 只在需要时添加详细信息
                if relation.get('details'):
                    connection['details'] = self._filter_important_details(
                        relation['details']
                    )
                
                connections.append(connection)
        
        return connections
    
    def _analyze_feature_pair(self, idx1: int, idx2: int) -> List[Dict[str, Any]]:
        """
        分析两个特征之间的拓扑关系
        
        Args:
            idx1: 第一个特征索引
            idx2: 第二个特征索引
            
        Returns:
            list: 关系列表
        """
        relations = []
        feature1 = self.features[idx1]
        feature2 = self.features[idx2]
        geom1 = self.feature_geometry_cache[idx1]
        geom2 = self.feature_geometry_cache[idx2]
        
        # 1. 检查相邻关系
        if self._check_adjacency(feature1.形状, feature2.形状):
            relations.append(self._create_relation(
                idx1, idx2, TopologyRelation.ADJACENT,
                {'shared_edges': self._count_shared_edges(feature1.形状, feature2.形状)}
            ))
        
        # 2. 检查几何关系（根据特征类型）
        if feature1.特征类型 == '孔' and feature2.特征类型 == '孔':
            # 检查孔之间的关系
            hole_relations = self._analyze_hole_relations(feature1, feature2, geom1, geom2)
            relations.extend(hole_relations)
            
        elif feature1.特征类型 == '平面' and feature2.特征类型 == '平面':
            # 检查平面之间的关系
            plane_relations = self._analyze_plane_relations(feature1, feature2, geom1, geom2)
            relations.extend(plane_relations)
            
        elif '圆柱' in feature1.特征类型 and '圆柱' in feature2.特征类型:
            # 检查圆柱面之间的关系
            cylinder_relations = self._analyze_cylinder_relations(feature1, feature2, geom1, geom2)
            relations.extend(cylinder_relations)
        
        # 3. 检查距离关系
        distance = self._calculate_minimum_distance(feature1.形状, feature2.形状)
        if distance is not None:
            if distance < self.tolerance:
                # 特征接触
                relations.append(self._create_relation(
                    idx1, idx2, "接触",
                    {'distance': distance}
                ))
            else:
                # 记录距离信息
                for relation in relations:
                    relation['details']['distance'] = distance
        
        return relations
    
    def _check_adjacency(self, shape1, shape2) -> bool:
        """
        检查两个形状是否相邻（共享边或顶点）
        
        Args:
            shape1: 第一个形状
            shape2: 第二个形状
            
        Returns:
            bool: 是否相邻
        """
        try:
            # 检查共享的边
            edges1 = set()
            explorer1 = TopExp_Explorer(shape1, TopAbs_EDGE)
            while explorer1.More():
                edge = topods.Edge(explorer1.Current())
                edges1.add(edge)
                explorer1.Next()
            
            explorer2 = TopExp_Explorer(shape2, TopAbs_EDGE)
            while explorer2.More():
                edge = topods.Edge(explorer2.Current())
                if edge in edges1:
                    return True
                explorer2.Next()
            
            # 检查共享的顶点
            vertices1 = set()
            explorer1 = TopExp_Explorer(shape1, TopAbs_VERTEX)
            while explorer1.More():
                vertex = topods.Vertex(explorer1.Current())
                pnt = BRep_Tool.Pnt(vertex)
                vertices1.add((round(pnt.X(), 6), round(pnt.Y(), 6), round(pnt.Z(), 6)))
                explorer1.Next()
            
            explorer2 = TopExp_Explorer(shape2, TopAbs_VERTEX)
            while explorer2.More():
                vertex = topods.Vertex(explorer2.Current())
                pnt = BRep_Tool.Pnt(vertex)
                vertex_coord = (round(pnt.X(), 6), round(pnt.Y(), 6), round(pnt.Z(), 6))
                if vertex_coord in vertices1:
                    return True
                explorer2.Next()
                
        except Exception as e:
            logger.warning(f"检查相邻关系时出错: {str(e)}")
        
        return False
    
    def _count_shared_edges(self, shape1, shape2) -> int:
        """计算共享边的数量"""
        count = 0
        try:
            edges1_map = TopTools_MapOfShape()
            explorer1 = TopExp_Explorer(shape1, TopAbs_EDGE)
            while explorer1.More():
                edges1_map.Add(explorer1.Current())
                explorer1.Next()
            
            explorer2 = TopExp_Explorer(shape2, TopAbs_EDGE)
            while explorer2.More():
                if edges1_map.Contains(explorer2.Current()):
                    count += 1
                explorer2.Next()
                
        except Exception as e:
            logger.warning(f"计算共享边时出错: {str(e)}")
        
        return count
    
    def _analyze_hole_relations(self, feature1, feature2, geom1, geom2) -> List[Dict[str, Any]]:
        """分析孔特征之间的关系"""
        relations = []
        
        # 获取孔的轴线信息
        axis1 = geom1.get('axis') or feature1.属性.get('轴向')
        axis2 = geom2.get('axis') or feature2.属性.get('轴向')
        loc1 = geom1.get('location') or feature1.属性.get('位置')
        loc2 = geom2.get('location') or feature2.属性.get('位置')
        
        if axis1 and axis2:
            # 检查同轴
            if self._are_vectors_parallel(axis1, axis2, self.tolerance):
                if self._are_points_on_same_line(loc1, loc2, axis1, self.tolerance):
                    relations.append(self._create_relation(
                        self.feature_index_map[id(feature1)],
                        self.feature_index_map[id(feature2)],
                        TopologyRelation.COAXIAL,
                        {'axis': axis1}
                    ))
                else:
                    # 平行但不同轴
                    relations.append(self._create_relation(
                        self.feature_index_map[id(feature1)],
                        self.feature_index_map[id(feature2)],
                        TopologyRelation.PARALLEL,
                        {'offset_distance': self._calculate_point_distance(loc1, loc2)}
                    ))
            
            # 检查垂直
            elif self._are_vectors_perpendicular(axis1, axis2, self.tolerance):
                relations.append(self._create_relation(
                    self.feature_index_map[id(feature1)],
                    self.feature_index_map[id(feature2)],
                    TopologyRelation.PERPENDICULAR,
                    {'angle': 90.0}
                ))
        
        return relations
    
    def _analyze_plane_relations(self, feature1, feature2, geom1, geom2) -> List[Dict[str, Any]]:
        """分析平面特征之间的关系"""
        relations = []
        
        normal1 = geom1.get('normal')
        normal2 = geom2.get('normal')
        loc1 = geom1.get('location')
        loc2 = geom2.get('location')
        
        if normal1 and normal2:
            # 检查共面
            if self._are_vectors_parallel(normal1, normal2, self.tolerance):
                if loc1 and loc2:
                    vec = (loc2[0] - loc1[0], loc2[1] - loc1[1], loc2[2] - loc1[2])
                    if abs(self._dot_product(vec, normal1)) < self.tolerance:
                        relations.append(self._create_relation(
                            self.feature_index_map[id(feature1)],
                            self.feature_index_map[id(feature2)],
                            TopologyRelation.COPLANAR,
                            {}
                        ))
                    else:
                        # 平行平面
                        distance = abs(self._dot_product(vec, normal1))
                        relations.append(self._create_relation(
                            self.feature_index_map[id(feature1)],
                            self.feature_index_map[id(feature2)],
                            TopologyRelation.PARALLEL,
                            {'distance': distance}
                        ))
            
            # 检查垂直
            elif self._are_vectors_perpendicular(normal1, normal2, self.tolerance):
                relations.append(self._create_relation(
                    self.feature_index_map[id(feature1)],
                    self.feature_index_map[id(feature2)],
                    TopologyRelation.PERPENDICULAR,
                    {'angle': 90.0}
                ))
        
        return relations
    
    def _analyze_cylinder_relations(self, feature1, feature2, geom1, geom2) -> List[Dict[str, Any]]:
        """分析圆柱面特征之间的关系"""
        relations = []
        
        axis1 = geom1.get('axis')
        axis2 = geom2.get('axis')
        loc1 = geom1.get('location')
        loc2 = geom2.get('location')
        radius1 = geom1.get('radius') or feature1.属性.get('半径')
        radius2 = geom2.get('radius') or feature2.属性.get('半径')
        
        if axis1 and axis2:
            # 检查同轴
            if self._are_vectors_parallel(axis1, axis2, self.tolerance):
                if loc1 and loc2 and self._are_points_on_same_line(loc1, loc2, axis1, self.tolerance):
                    relations.append(self._create_relation(
                        self.feature_index_map[id(feature1)],
                        self.feature_index_map[id(feature2)],
                        TopologyRelation.COAXIAL,
                        {'axis': axis1}
                    ))
                    
                    # 如果同轴，检查是否同心（半径相同）
                    if radius1 and radius2 and abs(radius1 - radius2) < self.tolerance:
                        relations.append(self._create_relation(
                            self.feature_index_map[id(feature1)],
                            self.feature_index_map[id(feature2)],
                            TopologyRelation.CONCENTRIC,
                            {'radius': radius1}
                        ))
        
        return relations
    
    def _calculate_minimum_distance(self, shape1, shape2) -> float:
        """计算两个形状之间的最小距离"""
        try:
            dist_shape_shape = BRepExtrema_DistShapeShape(shape1, shape2)
            if dist_shape_shape.IsDone():
                return dist_shape_shape.Value()
        except Exception as e:
            logger.warning(f"计算最小距离时出错: {str(e)}")
        return None
    
    def _are_vectors_parallel(self, v1: Tuple[float, float, float], 
                              v2: Tuple[float, float, float], 
                              tolerance: float) -> bool:
        """检查两个向量是否平行"""
        # 归一化向量
        norm1 = math.sqrt(sum(x**2 for x in v1))
        norm2 = math.sqrt(sum(x**2 for x in v2))
        
        if norm1 < tolerance or norm2 < tolerance:
            return False
        
        v1_norm = [x/norm1 for x in v1]
        v2_norm = [x/norm2 for x in v2]
        
        # 计算叉积
        cross = [
            v1_norm[1]*v2_norm[2] - v1_norm[2]*v2_norm[1],
            v1_norm[2]*v2_norm[0] - v1_norm[0]*v2_norm[2],
            v1_norm[0]*v2_norm[1] - v1_norm[1]*v2_norm[0]
        ]
        
        cross_magnitude = math.sqrt(sum(x**2 for x in cross))
        return cross_magnitude < tolerance
    
    def _are_vectors_perpendicular(self, v1: Tuple[float, float, float], 
                                   v2: Tuple[float, float, float], 
                                   tolerance: float) -> bool:
        """检查两个向量是否垂直"""
        dot = self._dot_product(v1, v2)
        return abs(dot) < tolerance
    
    def _dot_product(self, v1: Tuple[float, float, float], 
                    v2: Tuple[float, float, float]) -> float:
        """计算点积"""
        return v1[0]*v2[0] + v1[1]*v2[1] + v1[2]*v2[2]
    
    def _are_points_on_same_line(self, p1: Tuple[float, float, float],
                                 p2: Tuple[float, float, float],
                                 direction: Tuple[float, float, float],
                                 tolerance: float) -> bool:
        """检查两个点是否在同一条直线上"""
        # 标准化位置格式
        p1 = self._normalize_position(p1)
        p2 = self._normalize_position(p2)
        
        if p1 is None or p2 is None:
            return False
        
        if self._calculate_point_distance(p1, p2) < tolerance:
            return True
        
        # 计算p1到p2的向量
        vec = (p2[0] - p1[0], p2[1] - p1[1], p2[2] - p1[2])
        
        # 检查该向量是否与给定方向平行
        return self._are_vectors_parallel(vec, direction, tolerance)
    
    def _calculate_point_distance(self, p1: Tuple[float, float, float],
                                  p2: Tuple[float, float, float]) -> float:
        """计算两点之间的距离"""
        # 标准化位置格式
        p1 = self._normalize_position(p1)
        p2 = self._normalize_position(p2)
        return math.sqrt(sum((p2[i] - p1[i])**2 for i in range(3)))
    
    def _normalize_position(self, pos):
        """
        将位置数据标准化为元组格式
        支持输入: 
        - 元组/列表: (x, y, z)
        - 字典: {'x': x, 'y': y, 'z': z}
        """
        if pos is None:
            return None
        
        if isinstance(pos, dict):
            # 字典格式转元组
            return (pos.get('x', 0), pos.get('y', 0), pos.get('z', 0))
        elif isinstance(pos, (tuple, list)):
            # 已经是元组/列表格式
            return tuple(pos)
        else:
            # 其他格式，尝试转换
            try:
                return (float(pos[0]), float(pos[1]), float(pos[2]))
            except:
                return None
    
    def _create_relation(self, idx1: int, idx2: int, 
                        relation_type: str, 
                        details: Dict[str, Any]) -> Dict[str, Any]:
        """创建关系记录"""
        return {
            'feature1_index': idx1,
            'feature2_index': idx2,
            'feature1_type': self.features[idx1].特征类型,
            'feature2_type': self.features[idx2].特征类型,
            'relation_type': relation_type,
            'details': details
        }
    
    def get_feature_connections(self, feature_index: int) -> List[Dict[str, Any]]:
        """
        获取指定特征的所有连接关系
        
        Args:
            feature_index: 特征索引
            
        Returns:
            list: 该特征的所有关系（包含合并的多种关系类型）
        """
        connections = []
        for relation in self.topology_relations:
            if relation['feature1_index'] == feature_index or relation['feature2_index'] == feature_index:
                # 创建连接信息的副本，避免修改原始数据
                connection = relation.copy()
                
                # 如果有多个关系类型，展开显示
                if 'relation_type_list' in connection:
                    connection['relation_types'] = connection['relation_type_list']
                
                connections.append(connection)
        return connections
    
    def build_topology_graph(self) -> Dict[str, Any]:
        """
        构建拓扑关系图
        
        Returns:
            dict: 图结构数据
        """
        graph = {
            'nodes': [],
            'edges': []
        }
        
        # 添加节点（特征）
        for i, feature in enumerate(self.features):
            node = {
                'id': i,
                'type': feature.特征类型,
                'properties': feature.属性
            }
            graph['nodes'].append(node)
        
        # 添加边（关系），处理合并的多种关系
        for relation in self.topology_relations:
            edge = {
                'source': relation['feature1_index'],
                'target': relation['feature2_index'],
                'type': relation['relation_type'],  # 可能是多个类型的组合字符串
                'details': relation['details']
            }
            
            # 如果有多个关系类型，添加额外字段
            if 'relation_type_list' in relation:
                edge['types'] = relation['relation_type_list']
                edge['is_multi_relation'] = len(relation['relation_type_list']) > 1
            
            graph['edges'].append(edge)
        
        return graph
    
    def find_feature_patterns(self) -> List[Dict[str, Any]]:
        """
        查找特征模式（如阵列、对称等）
        
        Returns:
            list: 特征模式列表
        """
        patterns = []
        
        # 查找孔阵列
        hole_patterns = self._find_hole_arrays()
        patterns.extend(hole_patterns)
        
        # 查找对称特征
        symmetric_patterns = self._find_symmetric_features()
        patterns.extend(symmetric_patterns)
        
        return patterns
    
    def _find_hole_arrays(self) -> List[Dict[str, Any]]:
        """查找孔阵列模式"""
        patterns = []
        
        # 收集所有孔特征
        holes = [(i, f) for i, f in enumerate(self.features) if f.特征类型 == '孔']
        
        if len(holes) < 3:
            return patterns
        
        # 检查线性阵列
        for i in range(len(holes)):
            for j in range(i+1, len(holes)):
                for k in range(j+1, len(holes)):
                    idx1, hole1 = holes[i]
                    idx2, hole2 = holes[j]
                    idx3, hole3 = holes[k]
                    
                    loc1 = hole1.属性.get('位置')
                    loc2 = hole2.属性.get('位置')
                    loc3 = hole3.属性.get('位置')
                    
                    if loc1 and loc2 and loc3:
                        # 检查是否共线且等距
                        if self._check_linear_array(loc1, loc2, loc3):
                            # 查找该阵列的所有成员
                            array_members = self._find_all_array_members(holes, loc1, loc2)
                            if len(array_members) >= 3:
                                patterns.append({
                                    'type': '线性阵列',
                                    'feature_type': '孔',
                                    'members': array_members,
                                    'spacing': self._calculate_point_distance(loc1, loc2)
                                })
        
        return patterns
    
    def _check_linear_array(self, p1, p2, p3) -> bool:
        """检查三个点是否构成线性阵列（共线且等距）"""
        # 标准化位置格式
        p1 = self._normalize_position(p1)
        p2 = self._normalize_position(p2)
        p3 = self._normalize_position(p3)
        
        if p1 is None or p2 is None or p3 is None:
            return False
        
        # 检查共线
        v12 = (p2[0]-p1[0], p2[1]-p1[1], p2[2]-p1[2])
        v13 = (p3[0]-p1[0], p3[1]-p1[1], p3[2]-p1[2])
        
        if not self._are_vectors_parallel(v12, v13, 0.01):
            return False
        
        # 检查等距
        d12 = self._calculate_point_distance(p1, p2)
        d23 = self._calculate_point_distance(p2, p3)
        
        return abs(d12 - d23) < 0.1
    
    def _find_all_array_members(self, holes, p1, p2) -> List[int]:
        """查找阵列的所有成员"""
        # 标准化位置格式
        p1 = self._normalize_position(p1)
        p2 = self._normalize_position(p2)
        
        if p1 is None or p2 is None:
            return []
        
        direction = (p2[0]-p1[0], p2[1]-p1[1], p2[2]-p1[2])
        spacing = self._calculate_point_distance(p1, p2)
        
        members = []
        for idx, hole in holes:
            loc = self._normalize_position(hole.属性.get('位置'))
            if loc:
                # 检查是否在阵列线上
                for member_idx, member_hole in holes:
                    if member_idx != idx:
                        member_loc = self._normalize_position(member_hole.属性.get('位置'))
                        if member_loc:
                            vec = (loc[0]-member_loc[0], loc[1]-member_loc[1], loc[2]-member_loc[2])
                            if self._are_vectors_parallel(vec, direction, 0.01):
                                dist = self._calculate_point_distance(loc, member_loc)
                                if abs(dist % spacing) < 0.1:
                                    if idx not in members:
                                        members.append(idx)
                                    break
        
        return members
    
    def _find_symmetric_features(self) -> List[Dict[str, Any]]:
        """查找对称特征"""
        patterns = []
        
        # 简化实现：查找关于坐标平面对称的特征对
        for i in range(len(self.features)):
            for j in range(i+1, len(self.features)):
                if self.features[i].特征类型 == self.features[j].特征类型:
                    loc1 = self._normalize_position(self.features[i].属性.get('位置'))
                    loc2 = self._normalize_position(self.features[j].属性.get('位置'))
                    
                    if loc1 and loc2:
                        # 检查关于XY平面对称
                        if abs(loc1[0] - loc2[0]) < 0.1 and abs(loc1[1] - loc2[1]) < 0.1 and abs(loc1[2] + loc2[2]) < 0.1:
                            patterns.append({
                                'type': '镜像对称',
                                'plane': 'XY平面',
                                'features': [i, j]
                            })
                        # 检查关于XZ平面对称
                        elif abs(loc1[0] - loc2[0]) < 0.1 and abs(loc1[1] + loc2[1]) < 0.1 and abs(loc1[2] - loc2[2]) < 0.1:
                            patterns.append({
                                'type': '镜像对称',
                                'plane': 'XZ平面',
                                'features': [i, j]
                            })
                        # 检查关于YZ平面对称
                        elif abs(loc1[0] + loc2[0]) < 0.1 and abs(loc1[1] - loc2[1]) < 0.1 and abs(loc1[2] - loc2[2]) < 0.1:
                            patterns.append({
                                'type': '镜像对称',
                                'plane': 'YZ平面',
                                'features': [i, j]
                            })
        
        return patterns