#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
特征语义分析模块 - 新增
将基础几何特征组合为高级语义特征

功能：
1. 识别ISO简化螺纹表示（30°大半径锥面）
2. 组合内六角槽特征
3. 识别沉头孔/沉孔组合
4. 识别台阶孔
"""

import logging
import math
import numpy as np
from typing import Dict, List, Tuple, Optional, Any

from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import GeomAbs_Cone, GeomAbs_Cylinder, GeomAbs_Plane, GeomAbs_Circle
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.TopAbs import TopAbs_FACE, TopAbs_EDGE
from OCC.Core.TopoDS import topods
from OCC.Core.gp import gp_Pnt, gp_Vec
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop

logger = logging.getLogger(__name__)


# ISO标准螺纹参数
ISO_THREAD_STANDARDS = {
    # 公称直径: (螺距_粗牙, 小径)
    'M3': (0.5, 2.459),
    'M4': (0.7, 3.242),
    'M5': (0.8, 4.134),
    'M6': (1.0, 4.917),
    'M8': (1.25, 6.647),
    'M10': (1.5, 8.376),
    'M12': (1.75, 10.106),
    'M14': (2.0, 11.835),
    'M16': (2.0, 13.835),
    'M20': (2.5, 17.294),
    'M24': (3.0, 20.752),
}


class SemanticFeatureAnalyzer:
    """特征语义分析器"""
    
    # ISO简化螺纹锥面参数
    ISO_THREAD_CONE_ANGLE = 30.0  # 度
    ISO_THREAD_ANGLE_TOLERANCE = 5.0  # 容差
    ISO_THREAD_MIN_RADIUS_RATIO = 10.0  # 参考半径/公称直径的最小比值
    
    @staticmethod
    def analyze_semantic_features(features: List[Dict], shape) -> List[Dict]:
        """
        分析语义特征，将基础几何组合为高级特征
        
        Args:
            features: 基础特征列表
            shape: 整体形状
            
        Returns:
            List[Dict]: 识别到的语义特征
        """
        semantic_features = []
        
        try:
            # 1. 检查ISO简化螺纹
            thread_features = SemanticFeatureAnalyzer._find_iso_simplified_threads(features, shape)
            semantic_features.extend(thread_features)
            
            # 2. 检查沉头孔组合
            countersink_features = SemanticFeatureAnalyzer._find_countersink_holes(features, shape)
            semantic_features.extend(countersink_features)
            
            # 3. 检查台阶孔
            step_holes = SemanticFeatureAnalyzer._find_step_holes(features, shape)
            semantic_features.extend(step_holes)
            
            # 4. 检查凹端锥坑
            conical_pits = SemanticFeatureAnalyzer._find_conical_pits(features, shape)
            semantic_features.extend(conical_pits)
            
            return semantic_features
            
        except Exception as e:
            logger.error(f"语义特征分析时出错: {str(e)}")
            return semantic_features
    
    @staticmethod
    def _find_iso_simplified_threads(features: List[Dict], shape) -> List[Dict]:
        """
        识别ISO简化螺纹表示
        
        ISO/STEP标准中，螺纹常用30°大半径锥面简化表示
        特征：
        - 30°半角的锥面
        - 参考半径远大于螺纹公称直径（通常>500mm）
        - 位于圆柱面端部
        """
        thread_features = []
        
        try:
            # 找出所有锥面
            cone_faces = []
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while face_explorer.More():
                face = topods.Face(face_explorer.Current())
                surf = BRepAdaptor_Surface(face)
                
                if surf.GetType() == GeomAbs_Cone:
                    cone = surf.Cone()
                    semi_angle = math.degrees(cone.SemiAngle())
                    ref_radius = cone.RefRadius()
                    location = cone.Location()
                    axis = cone.Axis().Direction()
                    
                    cone_faces.append({
                        'face': face,
                        'semi_angle': semi_angle,
                        'ref_radius': ref_radius,
                        'location': (location.X(), location.Y(), location.Z()),
                        'axis': (axis.X(), axis.Y(), axis.Z())
                    })
                
                face_explorer.Next()
            
            # 检查每个锥面是否符合ISO简化螺纹特征
            for cone_data in cone_faces:
                angle = cone_data['semi_angle']
                radius = cone_data['ref_radius']
                
                # 检查角度是否接近30°
                if abs(angle - SemanticFeatureAnalyzer.ISO_THREAD_CONE_ANGLE) < SemanticFeatureAnalyzer.ISO_THREAD_ANGLE_TOLERANCE:
                    # 检查是否是大半径（简化表示特征）
                    if radius > 100:  # 大于100mm的参考半径
                        # 进一步验证：检查是否与圆柱面相邻
                        thread_info = SemanticFeatureAnalyzer._verify_iso_thread(cone_data, shape)
                        
                        if thread_info:
                            thread_features.append({
                                '类型': 'ISO简化螺纹',
                                '子类型': thread_info.get('thread_type', '外螺纹'),
                                '公称规格': thread_info.get('nominal_size', '未知'),
                                '位置': cone_data['location'],
                                '方向': cone_data['axis'],
                                '锥面参数': {
                                    '半角': round(angle, 1),
                                    '参考半径': round(radius, 1)
                                },
                                '置信度': thread_info.get('confidence', 0.7),
                                '原始面': cone_data['face']
                            })
            
            return thread_features
            
        except Exception as e:
            logger.warning(f"查找ISO简化螺纹时出错: {str(e)}")
            return []
    
    @staticmethod
    def _verify_iso_thread(cone_data: Dict, shape) -> Optional[Dict]:
        """
        验证锥面是否是ISO简化螺纹
        
        验证条件：
        1. 与圆柱面相邻
        2. 圆柱面直径在标准螺纹范围内
        """
        try:
            cone_face = cone_data['face']
            cone_axis = np.array(cone_data['axis'])
            
            # 查找与锥面相邻的圆柱面
            edge_explorer = TopExp_Explorer(cone_face, TopAbs_EDGE)
            
            while edge_explorer.More():
                edge = topods.Edge(edge_explorer.Current())
                
                # 查找共享这条边的其他面
                face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
                
                while face_explorer.More():
                    other_face = topods.Face(face_explorer.Current())
                    
                    if not other_face.IsSame(cone_face):
                        surf = BRepAdaptor_Surface(other_face)
                        
                        if surf.GetType() == GeomAbs_Cylinder:
                            # 检查是否共享边
                            other_edge_explorer = TopExp_Explorer(other_face, TopAbs_EDGE)
                            
                            while other_edge_explorer.More():
                                other_edge = topods.Edge(other_edge_explorer.Current())
                                
                                if edge.IsSame(other_edge):
                                    # 找到相邻圆柱面
                                    cylinder = surf.Cylinder()
                                    diameter = cylinder.Radius() * 2
                                    cyl_axis = cylinder.Axis().Direction()
                                    
                                    # 检查轴向是否一致
                                    cyl_axis_np = np.array([cyl_axis.X(), cyl_axis.Y(), cyl_axis.Z()])
                                    axis_dot = abs(np.dot(cone_axis, cyl_axis_np))
                                    
                                    if axis_dot > 0.99:  # 轴向一致
                                        # 匹配标准螺纹
                                        thread_size = SemanticFeatureAnalyzer._match_thread_size(diameter)
                                        
                                        if thread_size:
                                            return {
                                                'thread_type': '外螺纹',
                                                'nominal_size': thread_size,
                                                'diameter': diameter,
                                                'confidence': 0.85
                                            }
                                
                                other_edge_explorer.Next()
                    
                    face_explorer.Next()
                
                edge_explorer.Next()
            
            return None
            
        except Exception as e:
            logger.warning(f"验证ISO螺纹时出错: {str(e)}")
            return None
    
    @staticmethod
    def _match_thread_size(diameter: float) -> Optional[str]:
        """
        匹配标准螺纹尺寸
        """
        for thread_name, (pitch, minor_dia) in ISO_THREAD_STANDARDS.items():
            nominal_dia = float(thread_name[1:])  # 提取数字部分
            
            # 检查直径是否在公差范围内
            if abs(diameter - nominal_dia) < 0.3:
                return thread_name
        
        return None
    
    @staticmethod
    def _find_countersink_holes(features: List[Dict], shape) -> List[Dict]:
        """
        识别沉头孔组合
        
        沉头孔特征：
        - 圆柱孔 + 锥面入口
        - 锥面角度通常为82°、90°或120°
        """
        countersink_features = []
        
        try:
            # 标准沉头角度
            standard_angles = [41, 45, 60]  # 半角
            
            # 收集锥面和圆柱面
            cones = []
            cylinders = []
            
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            while face_explorer.More():
                face = topods.Face(face_explorer.Current())
                surf = BRepAdaptor_Surface(face)
                
                if surf.GetType() == GeomAbs_Cone:
                    cone = surf.Cone()
                    semi_angle = math.degrees(abs(cone.SemiAngle()))
                    
                    # 只考虑标准沉头角度
                    for std_angle in standard_angles:
                        if abs(semi_angle - std_angle) < 3:
                            cones.append({
                                'face': face,
                                'angle': semi_angle,
                                'location': cone.Location(),
                                'axis': cone.Axis().Direction(),
                                'radius': cone.RefRadius()
                            })
                            break
                
                elif surf.GetType() == GeomAbs_Cylinder:
                    cylinder = surf.Cylinder()
                    cylinders.append({
                        'face': face,
                        'radius': cylinder.Radius(),
                        'location': cylinder.Location(),
                        'axis': cylinder.Axis().Direction()
                    })
                
                face_explorer.Next()
            
            # 查找锥面和圆柱面的组合
            for cone in cones:
                for cyl in cylinders:
                    # 检查是否同轴
                    cone_axis = np.array([cone['axis'].X(), cone['axis'].Y(), cone['axis'].Z()])
                    cyl_axis = np.array([cyl['axis'].X(), cyl['axis'].Y(), cyl['axis'].Z()])
                    
                    if abs(np.dot(cone_axis, cyl_axis)) > 0.99:
                        # 检查是否相邻
                        if SemanticFeatureAnalyzer._are_faces_adjacent(cone['face'], cyl['face'], shape):
                            countersink_features.append({
                                '类型': '沉头孔',
                                '孔直径': round(cyl['radius'] * 2, 3),
                                '沉头角度': round(cone['angle'] * 2, 1),
                                '沉头直径': round(cone['radius'] * 2, 3),
                                '位置': (round(cyl['location'].X(), 3), round(cyl['location'].Y(), 3), round(cyl['location'].Z(), 3))
                            })
            
            return countersink_features
            
        except Exception as e:
            logger.warning(f"查找沉头孔时出错: {str(e)}")
            return []
    
    @staticmethod
    def _find_step_holes(features: List[Dict], shape) -> List[Dict]:
        """
        识别台阶孔
        
        台阶孔特征：
        - 多个同轴圆柱面
        - 直径不同
        """
        step_holes = []
        
        try:
            # 收集所有圆柱面
            cylinders = []
            
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            while face_explorer.More():
                face = topods.Face(face_explorer.Current())
                surf = BRepAdaptor_Surface(face)
                
                if surf.GetType() == GeomAbs_Cylinder:
                    cylinder = surf.Cylinder()
                    cylinders.append({
                        'face': face,
                        'radius': cylinder.Radius(),
                        'location': cylinder.Location(),
                        'axis': cylinder.Axis().Direction()
                    })
                
                face_explorer.Next()
            
            # 按轴向分组
            axis_groups = []
            
            for cyl in cylinders:
                found_group = False
                cyl_axis = np.array([cyl['axis'].X(), cyl['axis'].Y(), cyl['axis'].Z()])
                cyl_loc = np.array([cyl['location'].X(), cyl['location'].Y(), cyl['location'].Z()])
                
                for group in axis_groups:
                    ref_axis = group['axis']
                    ref_loc = group['location']
                    
                    # 检查轴向是否平行
                    if abs(np.dot(cyl_axis, ref_axis)) > 0.99:
                        # 检查是否在同一轴线上
                        loc_diff = cyl_loc - ref_loc
                        cross = np.cross(loc_diff, ref_axis)
                        if np.linalg.norm(cross) < 1.0:  # 1mm容差
                            group['cylinders'].append(cyl)
                            found_group = True
                            break
                
                if not found_group:
                    axis_groups.append({
                        'axis': cyl_axis,
                        'location': cyl_loc,
                        'cylinders': [cyl]
                    })
            
            # 识别台阶孔（同轴但不同直径）
            for group in axis_groups:
                if len(group['cylinders']) >= 2:
                    radii = [c['radius'] for c in group['cylinders']]
                    unique_radii = list(set([round(r, 2) for r in radii]))
                    
                    if len(unique_radii) >= 2:
                        step_holes.append({
                            '类型': '台阶孔',
                            '台阶数': len(unique_radii),
                            '直径列表': sorted([round(r * 2, 3) for r in unique_radii]),
                            '位置': (round(group['location'][0], 3), round(group['location'][1], 3), round(group['location'][2], 3))
                        })
            
            return step_holes
            
        except Exception as e:
            logger.warning(f"查找台阶孔时出错: {str(e)}")
            return []
    
    @staticmethod
    def _find_conical_pits(features: List[Dict], shape) -> List[Dict]:
        """
        识别凹端锥坑
        
        特征：
        - 小半径锥面（<10mm）
        - 45°或60°角度
        - 位于零件端部
        """
        conical_pits = []
        
        try:
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while face_explorer.More():
                face = topods.Face(face_explorer.Current())
                surf = BRepAdaptor_Surface(face)
                
                if surf.GetType() == GeomAbs_Cone:
                    cone = surf.Cone()
                    semi_angle = math.degrees(abs(cone.SemiAngle()))
                    ref_radius = cone.RefRadius()
                    location = cone.Location()
                    
                    # 小锥面且角度为45°或60°
                    if ref_radius < 10:
                        if abs(semi_angle - 45) < 5 or abs(semi_angle - 60) < 5:
                            conical_pits.append({
                                '类型': '凹端锥坑',
                                '锥角': round(semi_angle * 2, 1),
                                '半径': round(ref_radius, 3),
                                '位置': (round(location.X(), 3), round(location.Y(), 3), round(location.Z(), 3))
                            })
                
                face_explorer.Next()
            
            return conical_pits
            
        except Exception as e:
            logger.warning(f"查找凹端锥坑时出错: {str(e)}")
            return []
    
    @staticmethod
    def _are_faces_adjacent(face1, face2, shape) -> bool:
        """
        检查两个面是否相邻（共享边）
        """
        try:
            edge_explorer1 = TopExp_Explorer(face1, TopAbs_EDGE)
            
            while edge_explorer1.More():
                edge1 = topods.Edge(edge_explorer1.Current())
                
                edge_explorer2 = TopExp_Explorer(face2, TopAbs_EDGE)
                while edge_explorer2.More():
                    edge2 = topods.Edge(edge_explorer2.Current())
                    
                    if edge1.IsSame(edge2):
                        return True
                    
                    edge_explorer2.Next()
                
                edge_explorer1.Next()
            
            return False
            
        except:
            return False
    
    @staticmethod
    def analyze_cone_semantics(cone_face, shape) -> Dict[str, Any]:
        """
        分析单个锥面的语义含义
        
        Args:
            cone_face: 锥面
            shape: 整体形状
            
        Returns:
            Dict: 语义分析结果
        """
        result = {
            '原始类型': '圆锥面',
            '语义类型': '未知',
            '参数': {}
        }
        
        try:
            surf = BRepAdaptor_Surface(cone_face)
            if surf.GetType() != GeomAbs_Cone:
                return result
            
            cone = surf.Cone()
            semi_angle = math.degrees(abs(cone.SemiAngle()))
            ref_radius = cone.RefRadius()
            location = cone.Location()
            
            result['参数'] = {
                '半角': round(semi_angle, 1),
                '参考半径': round(ref_radius, 3),
                '位置': (round(location.X(), 3), round(location.Y(), 3), round(location.Z(), 3))
            }
            
            # 判断语义类型
            if ref_radius > 100 and abs(semi_angle - 30) < 5:
                result['语义类型'] = 'ISO简化螺纹'
            elif ref_radius < 10 and abs(semi_angle - 45) < 5:
                result['语义类型'] = '凹端锥坑'
            elif abs(semi_angle - 45) < 5:
                result['语义类型'] = '45°倒角'
            elif abs(semi_angle - 41) < 3:
                result['语义类型'] = '82°沉头孔'
            elif abs(semi_angle - 45) < 3:
                result['语义类型'] = '90°沉头孔'
            elif abs(semi_angle - 60) < 3:
                result['语义类型'] = '120°沉头孔'
            else:
                result['语义类型'] = '一般锥面'
            
            return result
            
        except Exception as e:
            logger.warning(f"分析锥面语义时出错: {str(e)}")
            result['错误'] = str(e)
            return result
