#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
机械特征识别模块，负责从STEP模型中识别机械特征
修复了孔被重复识别为圆柱面的问题
增加了STEP实体信息的记录功能
增加了拓扑关系分析功能
"""
import hashlib
import logging
from OCC.Core.TopAbs import TopAbs_FACE, TopAbs_EDGE
from OCC.Core.TopoDS import topods
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface
from OCC.Core.GeomAbs import GeomAbs_Cylinder, GeomAbs_Torus, GeomAbs_Plane, GeomAbs_Cone
from OCC.Extend.TopologyUtils import TopologyExplorer
from OCC.Core.STEPCAFControl import STEPCAFControl_Reader
from OCC.Core.TCollection import TCollection_ExtendedString
from OCC.Core.TDF import TDF_Label, TDF_Tool
from OCC.Core.TDataStd import TDataStd_Name
from OCC.Core.XCAFDoc import XCAFDoc_DocumentTool, XCAFDoc_ShapeTool
from OCC.Core.TopLoc import TopLoc_Location
from OCC.Core.BRep import BRep_Tool

from id2.feature_base import 机械特征, 特征类型
from id2.feature_analyzer import FeatureAnalyzer
from id2.feature_topology import FeatureTopologyAnalyzer  # 导入拓扑分析模块
import warnings
 
warnings.filterwarnings("ignore", category=DeprecationWarning)
logger = logging.getLogger(__name__)

class StepEntityMapper:
    """STEP实体映射器，用于管理形状与STEP实体之间的关系"""
    
    def __init__(self):
        self.shape_to_step_map = {}  # 形状到STEP实体的映射
        self.step_to_shape_map = {}  # STEP实体到形状的映射
        self.face_to_step_face_map = {}  # 面到STEP面实体的映射
        self.edge_to_step_edge_map = {}  # 边到STEP边实体的映射
        
    def add_shape_mapping(self, shape, step_entity_id, step_entity_type):
        """
        添加形状到STEP实体的映射
        
        Args:
            shape: OCC形状对象
            step_entity_id: STEP实体ID (如 #123)
            step_entity_type: STEP实体类型 (如 ADVANCED_FACE)
        """
        shape_id = id(shape)
        self.shape_to_step_map[shape_id] = {
            'step_id': step_entity_id,
            'step_type': step_entity_type,
            'shape': shape
        }
        self.step_to_shape_map[step_entity_id] = shape
        
    def add_face_mapping(self, face, step_face_id, step_surface_type):
        """
        添加面到STEP面实体的映射
        
        Args:
            face: OCC面对象
            step_face_id: STEP面实体ID
            step_surface_type: STEP表面类型 (如 CYLINDRICAL_SURFACE)
        """
        face_id = id(face)
        self.face_to_step_face_map[face_id] = {
            'step_id': step_face_id,
            'surface_type': step_surface_type,
            'face': face
        }
        
    def get_step_info(self, shape):
        """获取形状对应的STEP信息"""
        shape_id = id(shape)
        return self.shape_to_step_map.get(shape_id, None)
        
    def get_face_step_info(self, face):
        """获取面对应的STEP信息"""
        face_id = id(face)
        return self.face_to_step_face_map.get(face_id, None)

class FeatureRecognition:
    """特征识别类，负责从STEP模型中识别机械特征并分析拓扑关系"""

    def __init__(self, shapes, step_reader=None, step_mapper=None):
        """
        初始化特征识别器

        Args:
            shapes (list): OCC形状列表
            step_reader: STEP文件读取器（可选）
            step_mapper: STEP实体映射器（可选）
        """
        self.shapes = shapes
        self.features = []
        # 跟踪已处理过的面，使用更可靠的ID跟踪
        self.processed_faces = set()
        # 跟踪已识别为高级特征的面，避免重复识别为基本特征
        self.advanced_feature_faces = set()
        self._face_count = 0
        
        # STEP相关
        self.step_reader = step_reader
        self.step_mapper = step_mapper if step_mapper else StepEntityMapper()
        
        # 拓扑关系分析器
        self.topology_analyzer = None
        self.topology_relations = []
        self.feature_patterns = []
        
        # 如果提供了STEP读取器，初始化XCAF文档
        self.doc = None
        self.shape_tool = None
        if self.step_reader and hasattr(self.step_reader, 'GetDocument'):
            self.doc = self.step_reader.GetDocument()
            self.shape_tool = XCAFDoc_DocumentTool.ShapeTool(self.doc.Main())

    def _get_face_id(self, face):
        """
        获取面的唯一ID，使用更稳定的方法

        Args:
            face: OCC面对象

        Returns:
            str: 面的唯一ID
        """
        try:
            # 尝试使用面的几何信息生成更稳定的ID
            surf = BRepAdaptor_Surface(face)
            surf_type = surf.GetType()

            if surf_type == GeomAbs_Cylinder:
                cylinder = surf.Cylinder()
                location = cylinder.Location()
                radius = cylinder.Radius()
                direction = cylinder.Axis().Direction()

                # 使用位置、半径和方向信息生成ID
                data_str = f"cyl_{location.X():.10f}_{location.Y():.10f}_{location.Z():.10f}_{radius:.10f}_{direction.X():.10f}_{direction.Y():.10f}_{direction.Z():.10f}"
                hash_id = hashlib.md5(data_str.encode()).hexdigest()[:16]
                return hash_id

            elif surf_type == GeomAbs_Plane:
                plane = surf.Plane()
                location = plane.Location()
                normal = plane.Axis().Direction()

                data_str = f"plane_{location.X():.10f}_{location.Y():.10f}_{location.Z():.10f}_{normal.X():.10f}_{normal.Y():.10f}_{normal.Z():.10f}"
                hash_id = hashlib.md5(data_str.encode()).hexdigest()[:16]
                return hash_id

            elif surf_type == GeomAbs_Torus:
                torus = surf.Torus()
                location = torus.Location()
                major_radius = torus.MajorRadius()
                minor_radius = torus.MinorRadius()

                data_str = f"torus_{location.X():.10f}_{location.Y():.10f}_{location.Z():.10f}_{major_radius:.10f}_{minor_radius:.10f}"
                hash_id = hashlib.md5(data_str.encode()).hexdigest()[:16]
                return hash_id

            elif surf_type == GeomAbs_Cone:
                cone = surf.Cone()
                location = cone.Location()
                radius = cone.RefRadius()
                angle = cone.SemiAngle()

                data_str = f"cone_{location.X():.10f}_{location.Y():.10f}_{location.Z():.10f}_{radius:.10f}_{angle:.10f}"
                hash_id = hashlib.md5(data_str.encode()).hexdigest()[:16]
                return hash_id

            else:
                # 对于其他类型，使用内存地址作为备选
                return f"other_{id(face)}"

        except Exception as e:
            logger.warning(f"生成面ID时出错: {str(e)}")
            # 备选方案：使用内存地址
            return f"fallback_{id(face)}"

    def _record_step_info(self, feature, face):
        """
        记录特征的STEP信息
        
        Args:
            feature: 机械特征对象
            face: OCC面对象
        """
        # 从映射器获取STEP信息
        step_info = self.step_mapper.get_face_step_info(face)
        if step_info:
            feature.step_info = {
                'face_id': step_info['step_id'],
                'surface_type': step_info['surface_type']
            }
            
        # 如果有shape_tool，尝试获取更多信息
        if self.shape_tool:
            try:
                label = TDF_Label()
                if self.shape_tool.Search(face, label):
                    # 获取标签路径
                    entry = TCollection_ExtendedString()
                    TDF_Tool.Entry(label, entry)
                    feature.step_info['label_entry'] = entry.ToExtString()
                    
                    # 获取名称（如果有）
                    name_attr = TDataStd_Name()
                    if label.FindAttribute(TDataStd_Name.GetID(), name_attr):
                        feature.step_info['name'] = name_attr.Get().ToExtString()
            except Exception as e:
                logger.warning(f"获取STEP标签信息时出错: {str(e)}")

    def recognize_all_features(self, analyze_topology=True, manufacturing_only=True):
        """
        识别所有形状中的所有特征，并可选地分析拓扑关系

        Args:
            analyze_topology: 是否分析拓扑关系
            manufacturing_only: 是否只分析影响加工的拓扑关系

        Returns:
            dict: 包含特征列表和拓扑关系的字典
        """
        # 识别特征
        for shape in self.shapes:
            # 先识别高级特征（螺纹、孔等），因为这些特征通常包含多个面
            logger.debug("开始识别高级特征...")
            self._recognize_advanced_features(shape)

            # 再识别基本特征（平面、柱面、圆锥面等），避免与高级特征重复
            logger.debug("开始识别基本特征...")
            self._recognize_basic_features(shape)

        logger.info(f"总共识别到 {len(self.features)} 个特征")
        # ✅ 新增: 后处理验证和去重
        self.features = self._post_process_features(self.features)
    
        logger.info(f"后处理后剩余 {len(self.features)} 个特征")
        # 分析拓扑关系
        if analyze_topology and len(self.features) > 1:
            if manufacturing_only:
                logger.info("开始分析影响加工的拓扑关系...")
            else:
                logger.info("开始分析所有拓扑关系...")
            self.analyze_topology_relations(manufacturing_only=manufacturing_only)
        
        return {
            'features': self.features,
            'topology_relations': self.topology_relations,
            'feature_patterns': self.feature_patterns
        }

    def _post_process_features(self, features):
        """
        后处理: 去重、验证和过滤
        """
        logger.info(f"开始后处理 {len(features)} 个特征")
        
        # 1. 移除重复特征
        unique_features = self._remove_duplicates(features)
        logger.info(f"去重后剩余 {len(unique_features)} 个特征")
        
        # 2. 移除可能是基体的大面积"特征"
        filtered_features = self._filter_large_faces(unique_features)
        logger.info(f"过滤大面积后剩余 {len(filtered_features)} 个特征")
        
        # 3. 解决特征冲突
        resolved_features = self._resolve_conflicts(filtered_features)
        logger.info(f"冲突解决后剩余 {len(resolved_features)} 个特征")
        
        return resolved_features
    
    def _remove_duplicates(self, features):
        """移除重复特征"""
        unique = []
        seen_faces = set()
        
        for feature in features:
            face_id = self._get_face_id(feature.形状)
            
            if face_id not in seen_faces:
                unique.append(feature)
                seen_faces.add(face_id)
            else:
                logger.debug(f"移除重复特征: {feature.特征类型}")
        
        return unique
    
    def _filter_large_faces(self, features):
        """过滤可能是基体的大面积特征"""
        # 计算总面积
        total_area = 0.0
        for feature in features:
            try:
                props = GProp_GProps()
                brepgprop.SurfaceProperties(feature.形状, props)
                total_area += props.Mass()
            except:
                pass
        
        if total_area == 0:
            return features
        
        filtered = []
        for feature in features:
            try:
                props = GProp_GProps()
                brepgprop.SurfaceProperties(feature.形状, props)
                face_area = props.Mass()
                area_ratio = face_area / total_area
                
                # 凸台和凹槽不应该占总面积的15%以上
                if feature.特征类型 in ['凸台', '凹槽']:
                    if area_ratio > 0.15:
                        logger.warning(f"移除疑似基体的{feature.特征类型}, "
                                     f"面积比={area_ratio:.2%}")
                        continue
                
                filtered.append(feature)
                
            except Exception as e:
                logger.warning(f"过滤特征时出错: {str(e)}")
                filtered.append(feature)  # 保守处理
        
        return filtered
    
    def _resolve_conflicts(self, features):
        """解决特征冲突"""
        # 按置信度/面积排序
        sorted_features = sorted(
            features,
            key=lambda f: (
                f.属性.get('置信度', 0.5),  # 优先看置信度
                -f.属性.get('面积', 0)     # 其次看面积(负号=小面积优先)
            ),
            reverse=True
        )
        
        resolved = []
        used_faces = set()
        
        for feature in sorted_features:
            face_id = self._get_face_id(feature.形状)
            
            if face_id not in used_faces:
                resolved.append(feature)
                used_faces.add(face_id)
        
        return resolved

    def analyze_topology_relations(self, manufacturing_only=True):
        """
        分析特征之间的拓扑关系
        
        Args:
            manufacturing_only: 是否只分析影响加工的关系
        """
        if not self.features:
            logger.warning("没有特征可供分析拓扑关系")
            return
        
        # 创建拓扑分析器
        self.topology_analyzer = FeatureTopologyAnalyzer(
            self.features, 
            manufacturing_only=manufacturing_only
        )
        
        # 分析所有关系
        self.topology_relations = self.topology_analyzer.analyze_all_relations()
        
        if manufacturing_only:
            logger.info(f"识别到 {len(self.topology_relations)} 个影响加工的拓扑关系")
        else:
            logger.info(f"识别到 {len(self.topology_relations)} 个拓扑关系")
        
        # 查找特征模式
        self.feature_patterns = self.topology_analyzer.find_feature_patterns()
        logger.info(f"识别到 {len(self.feature_patterns)} 个特征模式")
        
        # 输出一些统计信息
        relation_types = {}
        
        for relation in self.topology_relations:
            # 统计关系类型
            if 'relation_type_list' in relation:
                for rel_type in relation['relation_type_list']:
                    relation_types[rel_type] = relation_types.get(rel_type, 0) + 1
            else:
                rel_type = relation['relation_type']
                relation_types[rel_type] = relation_types.get(rel_type, 0) + 1
        
        logger.info("拓扑关系统计:")
        for rel_type, count in relation_types.items():
            logger.info(f"  {rel_type}: {count}")
        
        # 输出特征模式信息
        if self.feature_patterns:
            logger.info("特征模式:")
            for pattern in self.feature_patterns:
                logger.info(f"  {pattern['type']}: {len(pattern.get('members', pattern.get('features', [])))} 个特征")

    def get_topology_graph(self):
        """
        获取拓扑关系图
        
        Returns:
            dict: 图结构数据
        """
        if self.topology_analyzer:
            return self.topology_analyzer.build_topology_graph()
        return None

    def get_feature_connections(self, feature_index):
        """
        获取指定特征的所有连接关系
        
        Args:
            feature_index: 特征索引
            
        Returns:
            list: 该特征的所有关系
        """
        if self.topology_analyzer:
            return self.topology_analyzer.get_feature_connections(feature_index)
        return []

    def _recognize_basic_features(self, shape):
        """
        识别基本特征（平面、柱面、圆锥面等）
        只识别未被高级特征占用的面

        Args:
            shape: OCC形状
        """
        explorer = TopExp_Explorer(shape, TopAbs_FACE)

        face_count = 0
        processed_count = 0
        skipped_count = 0

        while explorer.More():
            # 修复：使用新的静态方法
            face = topods.Face(explorer.Current())
            face_id = self._get_face_id(face)
            face_count += 1

            # 跳过已被高级特征处理过的面
            if face_id in self.advanced_feature_faces:
                skipped_count += 1
                logger.debug(f"跳过已被高级特征处理的面: {face_id[:50]}...")
                explorer.Next()
                continue

            # 跳过已处理过的面
            if face_id in self.processed_faces:
                skipped_count += 1
                explorer.Next()
                continue

            # 识别面的类型
            surface_type, properties = FeatureAnalyzer.analyze_surface(face)

            # 创建特征并添加到列表
            if surface_type != 特征类型['未知']:
                feature = 机械特征(surface_type, face, properties)
                
                # 记录STEP信息
                self._record_step_info(feature, face)
                
                self.features.append(feature)
                logger.debug(f"识别到基本特征: {surface_type} - {properties.get('类型', 'N/A')}")
                processed_count += 1

                # 标记为已处理
                self.processed_faces.add(face_id)

            explorer.Next()

        logger.debug(f"基本特征识别完成 - 总面数: {face_count}, 处理: {processed_count}, 跳过: {skipped_count}")

    def _recognize_advanced_features(self, shape):
        """
        识别高级特征（孔、槽、倒角等）

        Args:
            shape: OCC形状
        """
        # 按识别优先级排序，孔的优先级最高

        # 1. 识别螺纹（最高优先级）
        self._recognize_threads(shape)
        
        # 2. 识别孔（第二优先级）
        self._recognize_holes(shape)

        # 3. 识别圆角
        self._recognize_fillets(shape)

        # 4. 识别倒角
        self._recognize_chamfers(shape)

        # 5. 识别凹槽
        self._recognize_pockets(shape)

        # 6. 识别凸台
        self._recognize_bosses(shape)

        # 7. 识别槽
        self._recognize_slots(shape)

        # 8. 识别筋
        self._recognize_ribs(shape)

        # 9. 识别沟槽
        self._recognize_grooves(shape)

    def _recognize_holes(self, shape):
        """
        识别孔特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        hole_count = 0

        # 查找可能的孔（由圆柱面构成）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            if surf.GetType() == GeomAbs_Cylinder:
                # 检查是否为孔
                if FeatureAnalyzer.is_hole(face, shape):
                    # 分析孔的属性
                    properties = FeatureAnalyzer.analyze_hole(face, shape)

                    feature = 机械特征(特征类型['孔'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    # 如果是孔特征，可能需要记录相关的边界信息
                    if self.shape_tool:
                        self._record_hole_boundaries(feature, face, shape)
                    
                    self.features.append(feature)
                    hole_count += 1
                    logger.debug(
                        f"识别到孔特征 #{hole_count}: 直径={properties.get('直径', 'N/A')}, 深度={properties.get('深度', 'N/A')}")

                    # 关键：同时标记到两个集合中
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"孔特征识别完成，共识别到 {hole_count} 个孔")

    def _record_hole_boundaries(self, feature, face, shape):
        """
        记录孔特征的边界信息
        
        Args:
            feature: 机械特征对象
            face: 孔的圆柱面
            shape: 包含孔的形状
        """
        try:
            # 获取孔的边界边
            edge_explorer = TopExp_Explorer(face, TopAbs_EDGE)
            boundaries = []
            
            while edge_explorer.More():
                # 修复：使用新的静态方法
                edge = topods.Edge(edge_explorer.Current())
                edge_info = self.step_mapper.get_step_info(edge)
                if edge_info:
                    boundaries.append({
                        'edge_id': edge_info['step_id'],
                        'edge_type': edge_info['step_type']
                    })
                edge_explorer.Next()
                
            if boundaries:
                feature.step_info['boundaries'] = boundaries
                
        except Exception as e:
            logger.warning(f"记录孔边界信息时出错: {str(e)}")

    def _recognize_threads(self, shape):
        """
        识别螺纹特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        thread_count = 0

        # 查找可能的螺纹（由特殊的圆柱面构成）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            if surf.GetType() == GeomAbs_Cylinder:
                # 检查是否为螺纹
                if FeatureAnalyzer.is_thread(face, shape):
                    # 分析螺纹属性
                    properties = FeatureAnalyzer.analyze_thread(face, shape)

                    feature = 机械特征(特征类型['螺纹'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    self.features.append(feature)
                    thread_count += 1
                    logger.debug(
                        f"识别到螺纹特征 #{thread_count}: 规格={properties.get('规格', 'N/A')}, 类型={properties.get('类型', 'N/A')}")

                    # 关键：同时标记到两个集合中
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"螺纹特征识别完成，共识别到 {thread_count} 个螺纹")

    def _recognize_pockets(self, shape):
        """
        识别凹槽特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        pocket_count = 0

        # 查找可能的凹槽（通常底面是平面）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            if surf.GetType() == GeomAbs_Plane:
                # 检查是否为凹槽
                if FeatureAnalyzer.is_pocket(face, shape):
                    # 分析凹槽的属性
                    properties = FeatureAnalyzer.analyze_pocket(face, shape)

                    feature = 机械特征(特征类型['凹槽'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    self.features.append(feature)
                    pocket_count += 1
                    logger.debug(
                        f"识别到凹槽特征 #{pocket_count}: 深度={properties.get('深度', 'N/A')}, 底面形状={properties.get('底面形状', 'N/A')}")

                    # 标记为已处理
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"凹槽特征识别完成，共识别到 {pocket_count} 个凹槽")

    def _recognize_bosses(self, shape):
        """
        识别凸台特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        boss_count = 0

        # 查找可能的凸台（通常顶面是平面）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            if surf.GetType() == GeomAbs_Plane:
                # 检查是否为凸台
                if FeatureAnalyzer.is_boss(face, shape):
                    # 分析凸台的属性
                    properties = FeatureAnalyzer.analyze_boss(face, shape)

                    feature = 机械特征(特征类型['凸台'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    self.features.append(feature)
                    boss_count += 1
                    logger.debug(
                        f"识别到凸台特征 #{boss_count}: 高度={properties.get('高度', 'N/A')}, 顶面形状={properties.get('顶面形状', 'N/A')}")

                    # 标记为已处理
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"凸台特征识别完成，共识别到 {boss_count} 个凸台")

    def _recognize_slots(self, shape):
        """
        识别槽特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        slot_count = 0

        # 查找可能的槽（待实现具体逻辑）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            # TODO: 实现槽特征识别逻辑
            if surf.GetType() in [GeomAbs_Plane, GeomAbs_Cylinder]:
                # 检查是否为槽
                if FeatureAnalyzer.is_slot(face, shape):
                    # 分析槽的属性
                    properties = FeatureAnalyzer.analyze_slot(face, shape)

                    feature = 机械特征(特征类型['槽'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    self.features.append(feature)
                    slot_count += 1
                    logger.debug(f"识别到槽特征 #{slot_count}")

                    # 标记为已处理
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"槽特征识别完成，共识别到 {slot_count} 个槽")

    def _recognize_chamfers(self, shape):
        """
        识别倒角特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        chamfer_count = 0

        # 查找可能的倒角（通常是平面或锥面）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            if surf.GetType() in [GeomAbs_Plane, GeomAbs_Cone]:
                # 检查是否为倒角
                if FeatureAnalyzer.is_chamfer(face):
                    # 分析倒角属性
                    properties = FeatureAnalyzer.analyze_chamfer(face)

                    feature = 机械特征(特征类型['倒角'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    self.features.append(feature)
                    chamfer_count += 1
                    logger.debug(f"识别到倒角特征 #{chamfer_count}")

                    # 标记为已处理
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"倒角特征识别完成，共识别到 {chamfer_count} 个倒角")

    def _recognize_fillets(self, shape):
        """
        识别圆角特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        fillet_count = 0

        # 查找可能的圆角（由环面构成）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            if surf.GetType() == GeomAbs_Torus:
                # 检查是否为圆角
                if FeatureAnalyzer.is_fillet(face):
                    # 分析圆角属性
                    properties = FeatureAnalyzer.analyze_fillet(face)

                    feature = 机械特征(特征类型['圆角'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    self.features.append(feature)
                    fillet_count += 1
                    logger.debug(f"识别到圆角特征 #{fillet_count}: 半径={properties.get('半径', 'N/A')}")

                    # 标记为已处理
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"圆角特征识别完成，共识别到 {fillet_count} 个圆角")

    def _recognize_ribs(self, shape):
            """
            识别筋特征
    
            Args:
                shape: OCC形状
            """
            explorer = TopologyExplorer(shape)
            rib_count = 0
    
            # 查找可能的筋（通常由平面或柱面构成）
            for face in explorer.faces():
                face_id = self._get_face_id(face)
    
                # 跳过已处理过的面
                if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                    continue
    
                surf = BRepAdaptor_Surface(face)
    
                if surf.GetType() in [GeomAbs_Plane, GeomAbs_Cylinder]:
                    # 检查是否为筋
                    if FeatureAnalyzer.is_rib(face, shape):
                        # 分析筋的属性
                        properties = FeatureAnalyzer.analyze_rib(face, shape)
    
                        feature = 机械特征(特征类型['筋'], face, properties)
                        
                        # 记录STEP信息
                        self._record_step_info(feature, face)
                        
                        self.features.append(feature)
                        rib_count += 1
                        logger.debug(
                            f"识别到筋特征 #{rib_count}: 厚度={properties.get('厚度', 'N/A')}, 长度={properties.get('长度', 'N/A')}")
    
                        # 标记为已处理
                        self.processed_faces.add(face_id)
                        self.advanced_feature_faces.add(face_id)
    
            logger.debug(f"筋特征识别完成，共识别到 {rib_count} 个筋")
    
    def _recognize_grooves(self, shape):
        """
        识别沟槽特征

        Args:
            shape: OCC形状
        """
        explorer = TopologyExplorer(shape)
        groove_count = 0

        # 查找可能的沟槽（通常由平面或柱面构成的底面）
        for face in explorer.faces():
            face_id = self._get_face_id(face)

            # 跳过已处理过的面
            if face_id in self.processed_faces or face_id in self.advanced_feature_faces:
                continue

            surf = BRepAdaptor_Surface(face)

            if surf.GetType() in [GeomAbs_Plane, GeomAbs_Cylinder]:
                # 检查是否为沟槽
                if FeatureAnalyzer.is_groove(face, shape):
                    # 分析沟槽的属性
                    properties = FeatureAnalyzer.analyze_groove(face, shape)

                    feature = 机械特征(特征类型['沟槽'], face, properties)
                    
                    # 记录STEP信息
                    self._record_step_info(feature, face)
                    
                    self.features.append(feature)
                    groove_count += 1
                    logger.debug(
                        f"识别到沟槽特征 #{groove_count}: 长度={properties.get('长度', 'N/A')}, 深度={properties.get('深度', 'N/A')}")

                    # 标记为已处理
                    self.processed_faces.add(face_id)
                    self.advanced_feature_faces.add(face_id)

        logger.debug(f"沟槽特征识别完成，共识别到 {groove_count} 个沟槽")

    def get_feature_statistics(self):
        """
        获取特征识别统计信息

        Returns:
            dict: 特征统计信息
        """
        stats = {}
        for feature in self.features:
            feature_type = feature.特征类型
            if feature_type in stats:
                stats[feature_type] += 1
            else:
                stats[feature_type] = 1

        return stats

    def print_feature_summary(self):
        """
        打印特征识别摘要
        """
        stats = self.get_feature_statistics()
        logger.info("=== 特征识别摘要 ===")
        for feature_type, count in stats.items():
            logger.info(f"{feature_type}: {count} 个")
        logger.info(f"总计: {len(self.features)} 个特征")
        
        if self.topology_relations:
            logger.info(f"拓扑关系: {len(self.topology_relations)} 个")
        if self.feature_patterns:
            logger.info(f"特征模式: {len(self.feature_patterns)} 个")
        logger.info("===================")
        
    def get_features_with_step_info(self):
        """
        获取带有STEP信息的特征列表
        
        Returns:
            list: 包含STEP信息的特征列表
        """
        features_with_info = []
        for feature in self.features:
            if hasattr(feature, 'step_info') and feature.step_info:
                features_with_info.append({
                    'type': feature.特征类型,
                    'properties': feature.属性,
                    'step_info': feature.step_info
                })
        return features_with_info
        
    def export_feature_mapping(self, filename):
        """
        导出特征与STEP实体的映射关系，包括拓扑关系
        
        Args:
            filename: 输出文件名
        """
        import json
        
        mapping_data = {
            'features': [],
            'statistics': self.get_feature_statistics(),
            'topology_relations': [],
            'feature_patterns': []
        }
        
        # 导出特征信息
        for i, feature in enumerate(self.features):
            feature_data = {
                'id': i,
                'type': feature.特征类型,
                'properties': feature.属性
            }
            
            if hasattr(feature, 'step_info') and feature.step_info:
                feature_data['step_info'] = feature.step_info
                
            mapping_data['features'].append(feature_data)
        
        # 导出拓扑关系
        if self.topology_relations:
            for relation in self.topology_relations:
                mapping_data['topology_relations'].append({
                    'feature1': relation['feature1_index'],
                    'feature2': relation['feature2_index'],
                    'type': relation['relation_type'],
                    'details': relation['details']
                })
        
        # 导出特征模式
        if self.feature_patterns:
            mapping_data['feature_patterns'] = self.feature_patterns
        
        # 如果有拓扑分析器，导出拓扑图
        if self.topology_analyzer:
            mapping_data['topology_graph'] = self.topology_analyzer.build_topology_graph()
            
        with open(filename, 'w', encoding='utf-8') as f:
            json.dump(mapping_data, f, ensure_ascii=False, indent=2)
            
        logger.info(f"特征映射已导出到: {filename}")