#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
凸起/凹陷特征统一判断模块
解决凸台与凹槽的冲突问题
"""

import logging
import math
from typing import Tuple, Dict, List
from OCC.Core.gp import gp_Pnt, gp_Vec
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface
from OCC.Core.BRepClass3d import BRepClass3d_SolidClassifier
from OCC.Core.TopAbs import TopAbs_IN, TopAbs_OUT, TopAbs_ON
from OCC.Core.GeomAbs import GeomAbs_Plane

from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.TopAbs import TopAbs_FACE
from OCC.Core.TopoDS import topods
from OCC.Core.Precision import precision
logger = logging.getLogger(__name__)

class ProtrusionType:
    """特征突出类型"""
    PROTRUDING = "protruding"  # 凸起(向外)
    RECESSED = "recessed"      # 凹陷(向内)
    NEUTRAL = "neutral"        # 中性
    UNCERTAIN = "uncertain"    # 不确定

class FeatureProtrusionAnalyzer:
    """凸起/凹陷特征统一分析器 - 解决凸台与凹槽冲突"""
    
    # 配置参数
    MIN_TEST_OFFSET = 0.5      # 最小测试偏移(mm) - 增加以避免边界误判
    MAX_TEST_OFFSET = 3.0      # 最大测试偏移(mm)
    NUM_RADIAL_SAMPLES = 8     # 径向采样点数
    NUM_OFFSET_LEVELS = 3      # 偏移层级数
    CONFIDENCE_THRESHOLD = 0.75 # 置信度阈值
    MAX_FEATURE_AREA_RATIO = 0.15  # 特征面积不应超过总面积的15%
    
    @staticmethod
    def analyze_protrusion_type(face, shape) -> Tuple[str, float, Dict]:
        """
        统一分析面是凸起还是凹陷 - 多点采样增强版
        
        Args:
            face: 要分析的面
            shape: 整体形状
            
        Returns:
            Tuple[str, float, Dict]: (类型, 置信度, 详细信息)
        """
        result = {
            'type': ProtrusionType.UNCERTAIN,
            'confidence': 0.0,
            'outward_count': 0,
            'inward_count': 0,
            'boundary_count': 0,
            'test_points_total': 0,
            'face_area': 0.0,
            'total_area_ratio': 0.0,
            'is_large_face': False,
            'normal_direction': None
        }
        
        try:
            # 1. 检查是否为平面
            surf = BRepAdaptor_Surface(face)
            if surf.GetType() != GeomAbs_Plane:
                logger.debug("非平面，跳过凸起/凹陷判断")
                return ProtrusionType.NEUTRAL, 0.0, result
            
            # 2. 计算面积和面积比
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            face_area = props.Mass()
            face_center = props.CentreOfMass()
            result['face_area'] = face_area
            
            # 计算整体形状的总面积
            total_area = FeatureProtrusionAnalyzer._calculate_total_area(shape)
            if total_area > 0:
                area_ratio = face_area / total_area
                result['total_area_ratio'] = area_ratio
                
                # 如果面积比例过大，很可能是基体而非特征
                if area_ratio > FeatureProtrusionAnalyzer.MAX_FEATURE_AREA_RATIO:
                    result['is_large_face'] = True
                    logger.info(f"面积比例{area_ratio:.2%}超过阈值，可能是基体")
                    return ProtrusionType.NEUTRAL, 0.0, result
            
            # 3. 获取法向量
            plane = surf.Plane()
            normal = plane.Axis().Direction()
            normal_vec = gp_Vec(normal.X(), normal.Y(), normal.Z())
            
            # 确保法向量已归一化
            mag = normal_vec.Magnitude()
            if mag < 1e-10:
                logger.warning("法向量长度过小")
                return ProtrusionType.UNCERTAIN, 0.0, result
            normal_vec.Normalize()
            
            result['normal_direction'] = (normal_vec.X(), normal_vec.Y(), normal_vec.Z())
            
            # 4. 多层次多点采样
            outward_count = 0
            inward_count = 0
            boundary_count = 0
            test_points_total = 0
            
            # 在不同偏移距离上进行测试
            offset_distances = [
                FeatureProtrusionAnalyzer.MIN_TEST_OFFSET,
                (FeatureProtrusionAnalyzer.MIN_TEST_OFFSET + FeatureProtrusionAnalyzer.MAX_TEST_OFFSET) / 2,
                FeatureProtrusionAnalyzer.MAX_TEST_OFFSET
            ]
            
            for offset in offset_distances:
                # 在每个偏移层上进行径向采样
                for angle in [i * 2 * math.pi / FeatureProtrusionAnalyzer.NUM_RADIAL_SAMPLES 
                             for i in range(FeatureProtrusionAnalyzer.NUM_RADIAL_SAMPLES)]:
                    
                    # 创建测试点：中心 + 偏移法向量 + 径向扰动
                    # 径向扰动帮助覆盖整个面的区域
                    radial_offset = math.sqrt(face_area / math.pi) * 0.3  # 30%的半径
                    
                    # 创建径向扰动向量（在平面内）
                    # 使用法向量的两个正交向量
                    perp1 = gp_Vec(-normal_vec.Y(), normal_vec.X(), 0)
                    if perp1.Magnitude() < 1e-6:
                        perp1 = gp_Vec(1, 0, 0)
                    perp1.Normalize()
                    
                    perp2 = normal_vec.Crossed(perp1)
                    perp2.Normalize()
                    
                    radial_x = radial_offset * math.cos(angle)
                    radial_y = radial_offset * math.sin(angle)
                    
                    # 组合偏移
                    test_point = gp_Pnt(
                        face_center.X() + normal_vec.X() * offset + perp1.X() * radial_x + perp2.X() * radial_y,
                        face_center.Y() + normal_vec.Y() * offset + perp1.Y() * radial_x + perp2.Y() * radial_y,
                        face_center.Z() + normal_vec.Z() * offset + perp1.Z() * radial_x + perp2.Z() * radial_y
                    )
                    
                    # 测试点位置
                    classifier = BRepClass3d_SolidClassifier()
                    classifier.Load(shape)
                    classifier.Perform(test_point, precision.Confusion())
                    state = classifier.State()
                    
                    test_points_total += 1
                    
                    if state == TopAbs_OUT:
                        outward_count += 1
                    elif state == TopAbs_IN:
                        inward_count += 1
                    else:  # TopAbs_ON
                        boundary_count += 1
            
            result['outward_count'] = outward_count
            result['inward_count'] = inward_count
            result['boundary_count'] = boundary_count
            result['test_points_total'] = test_points_total
            
            # 5. 判断类型和置信度
            if test_points_total == 0:
                return ProtrusionType.UNCERTAIN, 0.0, result
            
            # 计算比例
            outward_ratio = outward_count / test_points_total
            inward_ratio = inward_count / test_points_total
            
            # 判断逻辑
            if outward_ratio > 0.7:  # 70%以上在外部 -> 凸起
                result_type = ProtrusionType.PROTRUDING
                confidence = outward_ratio
            elif inward_ratio > 0.7:  # 70%以上在内部 -> 凹陷
                result_type = ProtrusionType.RECESSED
                confidence = inward_ratio
            elif outward_ratio > inward_ratio and outward_ratio > 0.5:
                result_type = ProtrusionType.PROTRUDING
                confidence = outward_ratio * 0.8  # 降低置信度
            elif inward_ratio > outward_ratio and inward_ratio > 0.5:
                result_type = ProtrusionType.RECESSED
                confidence = inward_ratio * 0.8
            else:
                result_type = ProtrusionType.NEUTRAL
                confidence = 0.5
            
            result['type'] = result_type
            result['confidence'] = confidence
            
            logger.debug(f"凸起/凹陷分析: {result_type}, 置信度={confidence:.2f}, "
                        f"外部={outward_count}, 内部={inward_count}, 边界={boundary_count}")
            
            return result_type, confidence, result
            
        except Exception as e:
            logger.error(f"分析凸起/凹陷类型时出错: {str(e)}")
            return ProtrusionType.UNCERTAIN, 0.0, result
    
    @staticmethod
    def _calculate_total_area(shape) -> float:
        """计算整体形状的总表面积"""
        try:
            total_area = 0.0
            explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while explorer.More():
                face = topods.Face(explorer.Current())
                props = GProp_GProps()
                brepgprop.SurfaceProperties(face, props)
                total_area += props.Mass()
                explorer.Next()
            
            return total_area
        except Exception as e:
            logger.warning(f"计算总面积时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def is_boss_face(face, shape) -> Tuple[bool, float]:
        """
        判断是否为凸台面
        
        Returns:
            Tuple[bool, float]: (是否为凸台, 置信度)
        """
        protrusion_type, confidence, details = FeatureProtrusionAnalyzer.analyze_protrusion_type(face, shape)
        
        # 必须是凸起类型且置信度足够
        is_boss = (protrusion_type == ProtrusionType.PROTRUDING and 
                   confidence >= FeatureProtrusionAnalyzer.CONFIDENCE_THRESHOLD and
                   not details.get('is_large_face', False))
        
        return is_boss, confidence
    
    @staticmethod
    def is_pocket_face(face, shape) -> Tuple[bool, float]:
        """
        判断是否为凹槽面
        
        Returns:
            Tuple[bool, float]: (是否为凹槽, 置信度)
        """
        protrusion_type, confidence, details = FeatureProtrusionAnalyzer.analyze_protrusion_type(face, shape)
        
        # 必须是凹陷类型且置信度足够
        is_pocket = (protrusion_type == ProtrusionType.RECESSED and 
                     confidence >= FeatureProtrusionAnalyzer.CONFIDENCE_THRESHOLD and
                     not details.get('is_large_face', False))
        
        return is_pocket, confidence