#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
筋特征分析模块
筋的特征：薄壁结构，用于加强或连接，长度远大于厚度
"""

import logging
import math
from typing import Dict, List, Tuple, Optional
from OCC.Core.BRepBndLib import brepbndlib
from OCC.Core.TopAbs import TopAbs_FACE, TopAbs_EDGE, TopAbs_VERTEX
from OCC.Core.TopoDS import topods
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import GeomAbs_Plane, GeomAbs_Cylinder
from OCC.Extend.TopologyUtils import TopologyExplorer

from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop
from OCC.Core.BRep import BRep_Tool
from OCC.Core.gp import gp_Pnt, gp_Vec, gp_Dir, gp_Pln
from OCC.Core.GeomAPI import GeomAPI_ProjectPointOnSurf
from OCC.Core.BRepClass3d import BRepClass3d_SolidClassifier
from OCC.Core.TopAbs import TopAbs_IN, TopAbs_OUT, TopAbs_ON
logger = logging.getLogger(__name__)

class RibAnalyzer:
    """筋特征分析器"""
    
    @staticmethod
    def is_rib(face, shape) -> bool:
        """
        判断一个面是否属于筋特征
        筋的特点：
        1. 通常是平面或柱面
        2. 是薄壁结构，厚度相对较小
        3. 长度远大于厚度（长厚比 > 3）
        4. 通常用于连接或加强
        
        Args:
            face: 候选面
            shape: 整体形状
            
        Returns:
            bool: 是否为筋
        """
        try:
            surf = BRepAdaptor_Surface(face)
            
            # 筋通常由平面或柱面构成
            if surf.GetType() not in [GeomAbs_Plane, GeomAbs_Cylinder]:
                return False
            
            # 获取面的相邻面，分析整体结构
            adjacent_faces = RibAnalyzer._get_adjacent_faces(face, shape)
            
            if len(adjacent_faces) < 2:
                return False
            
            # 检查是否是薄壁结构
            thickness = RibAnalyzer._estimate_wall_thickness(face, adjacent_faces, shape)
            
            if thickness <= 0:
                return False
            
            # 计算面的主要尺寸
            dimensions = RibAnalyzer._calculate_face_dimensions(face)
            
            if not dimensions:
                return False
            
            length = max(dimensions['length'], dimensions['width'])
            
            # 筋的判断条件
            length_to_thickness_ratio = length / thickness if thickness > 0 else 0
            
            # 长厚比大于3，且厚度相对较小
            if length_to_thickness_ratio > 3.0 and thickness < length * 0.2:
                # 进一步检查是否是连接结构
                if RibAnalyzer._is_connecting_structure(face, adjacent_faces, shape):
                    return True
            
            return False
            
        except Exception as e:
            logger.warning(f"筋特征判断时出错: {str(e)}")
            return False
    
    @staticmethod
    def analyze_rib(face, shape) -> Dict:
        """
        分析筋特征的属性
        
        Args:
            face: 筋的面
            shape: 整体形状
            
        Returns:
            dict: 筋的属性
        """
        try:
            properties = {
                '类型': '筋',
                '厚度': 0.0,
                '长度': 0.0,
                '高度': 0.0,
                '长厚比': 0.0,
                '连接类型': '未知',
                '方向': None,
                '面积': 0.0
            }
            
            # 获取相邻面
            adjacent_faces = RibAnalyzer._get_adjacent_faces(face, shape)
            
            # 计算厚度
            thickness = RibAnalyzer._estimate_wall_thickness(face, adjacent_faces, shape)
            properties['厚度'] = round(thickness, 3)
            
            # 计算面的尺寸
            dimensions = RibAnalyzer._calculate_face_dimensions(face)
            if dimensions:
                properties['长度'] = round(dimensions['length'], 3)
                properties['高度'] = round(dimensions['width'], 3)
                properties['长厚比'] = round(dimensions['length'] / thickness, 2) if thickness > 0 else 0
            
            # 计算面积
            area = RibAnalyzer._calculate_surface_area(face)
            properties['面积'] = round(area, 3)
            
            # 分析连接类型
            connection_type = RibAnalyzer._analyze_connection_type(face, adjacent_faces, shape)
            properties['连接类型'] = connection_type
            
            # 计算主方向
            direction = RibAnalyzer._calculate_main_direction(face)
            if direction:
                properties['方向'] = (round(direction.X(), 3), round(direction.Y(), 3), round(direction.Z(), 3))
            
            return properties
            
        except Exception as e:
            logger.error(f"分析筋特征时出错: {str(e)}")
            return {'类型': '筋', '错误': str(e)}
    
    @staticmethod
    def _get_adjacent_faces(face, shape) -> List:
        """获取与给定面相邻的面"""
        adjacent_faces = []
        
        try:
            # 获取面的所有边
            edge_explorer = TopExp_Explorer(face, TopAbs_EDGE)
            face_edges = []
            
            while edge_explorer.More():
                edge = topods.Edge(edge_explorer.Current())
                face_edges.append(edge)
                edge_explorer.Next()
            
            # 对每条边，找到共享这条边的其他面
            for edge in face_edges:
                face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
                
                while face_explorer.More():
                    other_face = topods.Face(face_explorer.Current())
                    
                    if not face.IsSame(other_face):
                        # 检查other_face是否包含这条边
                        other_face_edge_explorer = TopExp_Explorer(other_face, TopAbs_EDGE)
                        
                        while other_face_edge_explorer.More():
                            other_edge = topods.Edge(other_face_edge_explorer.Current())
                            
                            if edge.IsSame(other_edge):
                                if other_face not in adjacent_faces:
                                    adjacent_faces.append(other_face)
                                break
                            
                            other_face_edge_explorer.Next()
                    
                    face_explorer.Next()
            
            return adjacent_faces
            
        except Exception as e:
            logger.warning(f"获取相邻面时出错: {str(e)}")
            return []
    
    @staticmethod
    def _estimate_wall_thickness(face, adjacent_faces, shape) -> float:
        """估算壁厚"""
        try:
            if not adjacent_faces:
                return 0.0
            
            # 获取面的中心点
            face_center = RibAnalyzer._get_face_center(face)
            
            if not face_center:
                return 0.0
            
            # 获取面的法向量
            surf = BRepAdaptor_Surface(face)
            u_mid = (surf.FirstUParameter() + surf.LastUParameter()) / 2
            v_mid = (surf.FirstVParameter() + surf.LastVParameter()) / 2
            
            pnt = gp_Pnt()
            vec_u = gp_Vec()
            vec_v = gp_Vec()
            surf.D1(u_mid, v_mid, pnt, vec_u, vec_v)
            
            normal = vec_u.Crossed(vec_v)
            if normal.Magnitude() > 0:
                normal.Normalize()
            else:
                return 0.0
            
            # 沿法向量方向寻找最近的对面
            min_distance = float('inf')
            
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                
                # 如果是平面且大致平行，计算距离
                if adj_surf.GetType() == GeomAbs_Plane and surf.GetType() == GeomAbs_Plane:
                    adj_plane = adj_surf.Plane()
                    face_plane = surf.Plane()
                    
                    # 检查是否平行
                    adj_normal = adj_plane.Axis().Direction()
                    face_normal = face_plane.Axis().Direction()
                    
                    dot_product = abs(adj_normal.Dot(face_normal))
                    
                    if dot_product > 0.9:  # 大致平行
                        distance = abs(face_plane.Distance(adj_plane.Location()))
                        min_distance = min(min_distance, distance)
            
            return min_distance if min_distance != float('inf') else 0.0
            
        except Exception as e:
            logger.warning(f"估算壁厚时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _calculate_face_dimensions(face) -> Optional[Dict]:
        """计算面的主要尺寸"""
        try:
            # 使用表面积属性计算
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            
            # 获取边界框估算尺寸
            bbox = RibAnalyzer._get_face_bounding_box(face)
            
            if not bbox:
                return None
            
            length = max(bbox['dx'], bbox['dy'], bbox['dz'])
            width = sorted([bbox['dx'], bbox['dy'], bbox['dz']])[1]  # 中间值
            
            return {
                'length': length,
                'width': width,
                'area': props.Mass()
            }
            
        except Exception as e:
            logger.warning(f"计算面尺寸时出错: {str(e)}")
            return None
    
    @staticmethod
    def _get_face_bounding_box(face) -> Optional[Dict]:
        """获取面的边界框"""
        try:
            from OCC.Core.Bnd import Bnd_Box
            bbox = Bnd_Box()
            brepbndlib.Add(face, bbox)
            
            xmin, ymin, zmin, xmax, ymax, zmax = bbox.Get()
            
            return {
                'xmin': xmin, 'ymin': ymin, 'zmin': zmin,
                'xmax': xmax, 'ymax': ymax, 'zmax': zmax,
                'dx': xmax - xmin,
                'dy': ymax - ymin,
                'dz': zmax - zmin
            }
            
        except Exception as e:
            logger.warning(f"获取边界框时出错: {str(e)}")
            return None
    
    @staticmethod
    def _is_connecting_structure(face, adjacent_faces, shape) -> bool:
        """判断是否是连接结构"""
        try:
            # 简化的判断：如果有多个相邻面且分布在不同方向，可能是连接结构
            if len(adjacent_faces) >= 2:
                return True
            
            return False
            
        except Exception as e:
            logger.warning(f"判断连接结构时出错: {str(e)}")
            return False
    
    @staticmethod
    def _calculate_surface_area(face) -> float:
        """计算面的表面积"""
        try:
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            return props.Mass()
        except Exception as e:
            logger.warning(f"计算面积时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _get_face_center(face) -> Optional[gp_Pnt]:
        """获取面的中心点"""
        try:
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            return props.CentreOfMass()
        except Exception as e:
            logger.warning(f"获取面中心时出错: {str(e)}")
            return None
    
    @staticmethod
    def _calculate_main_direction(face) -> Optional[gp_Dir]:
        """计算面的主方向"""
        try:
            surf = BRepAdaptor_Surface(face)
            
            if surf.GetType() == GeomAbs_Plane:
                plane = surf.Plane()
                return plane.Axis().Direction()
            elif surf.GetType() == GeomAbs_Cylinder:
                cylinder = surf.Cylinder()
                return cylinder.Axis().Direction()
            
            return None
            
        except Exception as e:
            logger.warning(f"计算主方向时出错: {str(e)}")
            return None
    
    @staticmethod
    def _analyze_connection_type(face, adjacent_faces, shape) -> str:
        """分析连接类型"""
        try:
            if len(adjacent_faces) >= 4:
                return "多面连接"
            elif len(adjacent_faces) == 3:
                return "三面连接"
            elif len(adjacent_faces) == 2:
                return "双面连接"
            else:
                return "单面连接"
                
        except Exception as e:
            logger.warning(f"分析连接类型时出错: {str(e)}")
            return "未知"