#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
凹槽特征分析模块
凹槽的特征：封闭或半封闭的凹陷区域，通常用于减重、容纳或定位
"""

import logging
import math
from typing import Dict, List, Tuple, Optional

from OCC.Core.TopAbs import TopAbs_FACE, TopAbs_EDGE, TopAbs_VERTEX
from OCC.Core.TopoDS import topods
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import GeomAbs_Plane, GeomAbs_Cylinder, GeomAbs_Cone
from OCC.Extend.TopologyUtils import TopologyExplorer
from OCC.Core.Bnd import Bnd_Box
from OCC.Core.BRepBndLib import brepbndlib
from OCC.Core.Precision import precision
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop
from OCC.Core.BRep import BRep_Tool
from OCC.Core.gp import gp_Pnt, gp_Vec, gp_Dir, gp_Pln
from OCC.Core.GeomAPI import GeomAPI_ProjectPointOnSurf
from OCC.Core.BRepClass3d import BRepClass3d_SolidClassifier
from OCC.Core.TopAbs import TopAbs_IN, TopAbs_OUT, TopAbs_ON
# 导入统一判断模块
from id2.feature_protrusion_analyzer import FeatureProtrusionAnalyzer

logger = logging.getLogger(__name__)

class PocketAnalyzer:
    """凹槽特征分析器"""
    
    @staticmethod
    def is_pocket(face, shape) -> bool:
        """
        判断一个面是否属于凹槽特征 - 修复版
        
        改进:
        1. 使用统一的凸起/凹陷判断
        2. 增加面积阈值检查
        3. 增加封闭性检查
        """
        try:
            surf = BRepAdaptor_Surface(face)
            
            # 凹槽的底面通常是平面
            if surf.GetType() != GeomAbs_Plane:
                return False
            
            # ✅ 修复1: 使用统一的凹陷判断
            is_recessed, confidence = FeatureProtrusionAnalyzer.is_pocket_face(face, shape)
            
            if not is_recessed:
                logger.debug(f"不是凹陷特征或置信度不足: {confidence:.2f}")
                return False
            
            # 获取相邻面分析整体结构
            adjacent_faces = PocketAnalyzer._get_adjacent_faces(face, shape)
            
            if len(adjacent_faces) < 3:  # 凹槽至少需要3个侧壁
                return False
            
            # ✅ 修复2: 检查是否是封闭或半封闭结构
            if not PocketAnalyzer._is_enclosed_cavity(face, adjacent_faces, shape):
                return False
            
            # 分析几何特征
            geometry_analysis = PocketAnalyzer._analyze_pocket_geometry(face, adjacent_faces, shape)
            
            if not geometry_analysis:
                return False
            
            # ✅ 修复3: 增加尺寸合理性检查
            depth = geometry_analysis.get('depth', 0)
            opening_area = geometry_analysis.get('opening_area', 0)
            bottom_area = geometry_analysis.get('bottom_area', 0)
            
            # 有合理的深度，且不是贯通的
            if depth > 0 and bottom_area > 0:
                characteristic_length = math.sqrt(bottom_area)
                
                # ✅ 修复4: 深度约束
                if not (characteristic_length * 0.1 < depth < characteristic_length * 3):
                    logger.debug(f"深度不合理: d={depth:.2f}, L={characteristic_length:.2f}")
                    return False
                
                # 检查是否贯通
                if PocketAnalyzer._is_through_feature(face, shape):
                    logger.debug("贯通特征不是凹槽")
                    return False
                
                # 检查是否有典型的凹槽侧壁
                if PocketAnalyzer._has_pocket_sidewalls(face, adjacent_faces):
                    return True
            
            return False
            
        except Exception as e:
            logger.warning(f"凹槽特征判断时出错: {str(e)}")
            return False
    
    @staticmethod
    def analyze_pocket(face, shape) -> Dict:
        """
        分析凹槽特征的属性
        
        Args:
            face: 凹槽的底面
            shape: 整体形状
            
        Returns:
            dict: 凹槽的属性
        """
        try:
            properties = {
                '类型': '凹槽',
                '深度': 0.0,
                '底面形状': '未知',
                '底面尺寸': {},
                '开口尺寸': {},
                '锥度': 0.0,
                '底面积': 0.0,
                '开口面积': 0.0,
                '体积': 0.0,
                '侧壁数量': 0,
                '封闭类型': '未知',
                '用途': '未知'
            }
            
            # 获取相邻面
            adjacent_faces = PocketAnalyzer._get_adjacent_faces(face, shape)
            properties['侧壁数量'] = len(adjacent_faces)
            
            # 分析几何特征
            geometry = PocketAnalyzer._analyze_pocket_geometry(face, adjacent_faces, shape)
            
            if geometry:
                properties['深度'] = round(geometry['depth'], 3)
                properties['底面积'] = round(geometry['bottom_area'], 3)
                properties['开口面积'] = round(geometry['opening_area'], 3)
                properties['锥度'] = round(geometry.get('taper', 0), 3)
            
            # 分析底面形状
            bottom_shape = PocketAnalyzer._analyze_face_shape(face)
            properties['底面形状'] = bottom_shape['shape']
            properties['底面尺寸'] = bottom_shape['dimensions']
            
            # 估算开口尺寸
            opening_dimensions = PocketAnalyzer._estimate_opening_dimensions(face, adjacent_faces, geometry)
            properties['开口尺寸'] = opening_dimensions
            
            # 估算体积
            volume = PocketAnalyzer._estimate_pocket_volume(face, adjacent_faces, geometry)
            properties['体积'] = round(volume, 3)
            
            # 分析封闭类型
            closure_type = PocketAnalyzer._analyze_closure_type(face, adjacent_faces)
            properties['封闭类型'] = closure_type
            
            # 分析可能用途
            purpose = PocketAnalyzer._analyze_pocket_purpose(face, adjacent_faces, geometry)
            properties['用途'] = purpose
            
            return properties
            
        except Exception as e:
            logger.error(f"分析凹槽特征时出错: {str(e)}")
            return {'类型': '凹槽', '错误': str(e)}
    
    @staticmethod
    def _get_adjacent_faces(face, shape) -> List:
        """获取与给定面相邻的面"""
        adjacent_faces = []
        
        try:
            # 获取面的所有边
            edge_explorer = TopExp_Explorer(face, TopAbs_EDGE)
            face_edges = []
            
            while edge_explorer.More():
                edge = topods.Edge(edge_explorer.Current())
                face_edges.append(edge)
                edge_explorer.Next()
            
            # 对每条边，找到共享这条边的其他面
            for edge in face_edges:
                face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
                
                while face_explorer.More():
                    other_face = topods.Face(face_explorer.Current())
                    
                    if not face.IsSame(other_face):
                        # 检查other_face是否包含这条边
                        other_face_edge_explorer = TopExp_Explorer(other_face, TopAbs_EDGE)
                        
                        while other_face_edge_explorer.More():
                            other_edge = topods.Edge(other_face_edge_explorer.Current())
                            
                            if edge.IsSame(other_edge):
                                if other_face not in adjacent_faces:
                                    adjacent_faces.append(other_face)
                                break
                            
                            other_face_edge_explorer.Next()
                    
                    face_explorer.Next()
            
            return adjacent_faces
            
        except Exception as e:
            logger.warning(f"获取相邻面时出错: {str(e)}")
            return []
    
    @staticmethod
    def _is_recessed_feature(face, shape) -> bool:
        """判断是否是凹陷特征（向内的）"""
        try:
            # 获取面的中心点和法向量
            face_center = PocketAnalyzer._get_face_center(face)
            if not face_center:
                return False
            
            surf = BRepAdaptor_Surface(face)
            u_mid = (surf.FirstUParameter() + surf.LastUParameter()) / 2
            v_mid = (surf.FirstVParameter() + surf.LastVParameter()) / 2
            
            pnt = gp_Pnt()
            vec_u = gp_Vec()
            vec_v = gp_Vec()
            surf.D1(u_mid, v_mid, pnt, vec_u, vec_v)
            
            normal = vec_u.Crossed(vec_v)
            if normal.Magnitude() > 0:
                normal.Normalize()
            else:
                return False
            
            # 沿法向量方向检查是否指向实体内部
            test_point = gp_Pnt(
                face_center.X() + normal.X() * 0.01,
                face_center.Y() + normal.Y() * 0.01,
                face_center.Z() + normal.Z() * 0.01
            )
            
            classifier = BRepClass3d_SolidClassifier()
            classifier.Load(shape)
            classifier.Perform(test_point, precision.Confusion())
            
            # 如果测试点在实体内部，说明法向量指向内部，是凹陷特征
            return classifier.State() == TopAbs_IN
            
        except Exception as e:
            logger.warning(f"判断凹陷特征时出错: {str(e)}")
            return False
    
    @staticmethod
    def _is_enclosed_cavity(face, adjacent_faces, shape) -> bool:
        """判断是否是封闭或半封闭的空腔"""
        try:
            if len(adjacent_faces) < 3:
                return False
            
            # 检查相邻面是否形成围墙结构
            face_center = PocketAnalyzer._get_face_center(face)
            if not face_center:
                return False
            
            # 计算相邻面的法向量，看是否大致指向中心
            inward_faces = 0
            
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                
                if adj_surf.GetType() in [GeomAbs_Plane, GeomAbs_Cylinder]:
                    adj_center = PocketAnalyzer._get_face_center(adj_face)
                    if adj_center:
                        # 计算从相邻面中心到底面中心的向量
                        to_center = gp_Vec(adj_center, face_center)
                        
                        if adj_surf.GetType() == GeomAbs_Plane:
                            adj_normal = adj_surf.Plane().Axis().Direction()
                            
                            # 检查法向量是否大致指向底面
                            if to_center.Magnitude() > 0:
                                to_center.Normalize()
                                dot_product = adj_normal.Dot(gp_Dir(to_center))
                                
                                if abs(dot_product) > 0.5:  # 大致朝向或背离中心
                                    inward_faces += 1
            
            # 如果大部分相邻面都朝向或背离中心，认为是封闭结构
            return inward_faces >= len(adjacent_faces) * 0.6
            
        except Exception as e:
            logger.warning(f"判断封闭空腔时出错: {str(e)}")
            return False
    
    @staticmethod
    def _is_through_feature(face, shape) -> bool:
        """判断是否是贯通特征"""
        try:
            # 简化判断：检查是否有对应的出口面
            surf = BRepAdaptor_Surface(face)
            face_normal = surf.Plane().Axis().Direction()
            face_center = PocketAnalyzer._get_face_center(face)
            
            if not face_center:
                return False
            
            # 在相反方向寻找平行面
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while face_explorer.More():
                other_face = topods.Face(face_explorer.Current())
                
                if not face.IsSame(other_face):
                    other_surf = BRepAdaptor_Surface(other_face)
                    
                    if other_surf.GetType() == GeomAbs_Plane:
                        other_normal = other_surf.Plane().Axis().Direction()
                        other_center = PocketAnalyzer._get_face_center(other_face)
                        
                        if other_center:
                            # 检查是否平行且方向相反
                            dot_product = face_normal.Dot(other_normal)
                            
                            if dot_product < -0.8:  # 反向平行
                                distance = face_center.Distance(other_center)
                                
                                # 如果距离合理，可能是贯通的
                                if distance > 0.1:  # 最小厚度阈值
                                    return True
                
                face_explorer.Next()
            
            return False
            
        except Exception as e:
            logger.warning(f"判断贯通特征时出错: {str(e)}")
            return False
    
    @staticmethod
    def _analyze_pocket_geometry(face, adjacent_faces, shape) -> Optional[Dict]:
        """分析凹槽的几何特征"""
        try:
            # 计算底面面积
            bottom_area = PocketAnalyzer._calculate_surface_area(face)
            
            # 估算深度和开口面积
            depth, opening_area = PocketAnalyzer._estimate_depth_and_opening_area(face, adjacent_faces, shape)
            
            # 计算锥度（如果有）
            taper = PocketAnalyzer._calculate_taper(face, adjacent_faces, depth)
            
            return {
                'depth': depth,
                'bottom_area': bottom_area,
                'opening_area': opening_area,
                'taper': taper
            }
            
        except Exception as e:
            logger.warning(f"分析凹槽几何时出错: {str(e)}")
            return None
    
    @staticmethod
    def _estimate_depth_and_opening_area(face, adjacent_faces, shape) -> Tuple[float, float]:
        """估算凹槽深度和开口面积"""
        try:
            if not adjacent_faces:
                return 0.0, 0.0
            
            face_center = PocketAnalyzer._get_face_center(face)
            if not face_center:
                return 0.0, 0.0
            
            # 寻找可能的开口面（通常是最远的平行面）
            surf = BRepAdaptor_Surface(face)
            face_normal = surf.Plane().Axis().Direction()
            
            max_distance = 0.0
            opening_face = None
            
            # 在整个形状中寻找可能的开口面
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while face_explorer.More():
                other_face = topods.Face(face_explorer.Current())
                
                if not face.IsSame(other_face):
                    other_surf = BRepAdaptor_Surface(other_face)
                    
                    if other_surf.GetType() == GeomAbs_Plane:
                        other_normal = other_surf.Plane().Axis().Direction()
                        other_center = PocketAnalyzer._get_face_center(other_face)
                        
                        if other_center:
                            # 检查是否平行且方向相反（开口面）
                            dot_product = face_normal.Dot(other_normal)
                            
                            if dot_product < -0.7:  # 大致反向平行
                                distance = face_center.Distance(other_center)
                                
                                if distance > max_distance:
                                    max_distance = distance
                                    opening_face = other_face
                
                face_explorer.Next()
            
            # 计算开口面积
            opening_area = 0.0
            if opening_face:
                opening_area = PocketAnalyzer._calculate_surface_area(opening_face)
            else:
                # 如果找不到明确的开口面，估算一个开口面积
                bottom_area = PocketAnalyzer._calculate_surface_area(face)
                opening_area = bottom_area * 1.5  # 假设开口比底面大50%
            
            return max_distance, opening_area
            
        except Exception as e:
            logger.warning(f"估算深度和开口面积时出错: {str(e)}")
            return 0.0, 0.0
    
    @staticmethod
    def _calculate_taper(face, adjacent_faces, depth) -> float:
        """计算锥度"""
        try:
            if not adjacent_faces or depth <= 0:
                return 0.0
            
            # 寻找锥形侧面
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                
                if adj_surf.GetType() == GeomAbs_Cone:
                    cone = adj_surf.Cone()
                    semi_angle = cone.SemiAngle()
                    return math.degrees(semi_angle)
            
            return 0.0  # 直壁凹槽
            
        except Exception as e:
            logger.warning(f"计算锥度时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _has_pocket_sidewalls(face, adjacent_faces) -> bool:
        """检查是否有典型的凹槽侧壁"""
        try:
            if len(adjacent_faces) < 3:
                return False
            
            # 检查相邻面是否有垂直或倾斜的侧壁
            surf = BRepAdaptor_Surface(face)
            face_normal = surf.Plane().Axis().Direction()
            
            sidewall_count = 0
            
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                
                if adj_surf.GetType() in [GeomAbs_Plane, GeomAbs_Cylinder, GeomAbs_Cone]:
                    if adj_surf.GetType() == GeomAbs_Plane:
                        adj_normal = adj_surf.Plane().Axis().Direction()
                        
                        # 检查是否垂直或接近垂直
                        dot_product = abs(face_normal.Dot(adj_normal))
                        
                        if dot_product < 0.3:  # 接近垂直
                            sidewall_count += 1
                    else:
                        sidewall_count += 1
            
            # 至少有3个侧壁
            return sidewall_count >= 3
            
        except Exception as e:
            logger.warning(f"检查侧壁时出错: {str(e)}")
            return False
    
    @staticmethod
    def _analyze_face_shape(face) -> Dict:
        """分析面的形状"""
        try:
            result = {
                'shape': '未知',
                'dimensions': {}
            }
            
            # 获取边界框
            bbox = PocketAnalyzer._get_face_bounding_box(face)
            if not bbox:
                return result
            
            dx, dy, dz = bbox['dx'], bbox['dy'], bbox['dz']
            
            # 找到主要的两个维度（忽略最小的）
            dimensions = [dx, dy, dz]
            dimensions.sort(reverse=True)
            
            dim1, dim2 = dimensions[0], dimensions[1]
            
            # 判断形状
            ratio = dim1 / dim2 if dim2 > 0 else float('inf')
            
            if ratio < 1.2:  # 接近正方形或圆形
                result['shape'] = '圆形或正方形'
                result['dimensions'] = {'直径/边长': round(dim1, 3)}
            elif ratio < 2.0:  # 矩形
                result['shape'] = '矩形'
                result['dimensions'] = {
                    '长度': round(dim1, 3),
                    '宽度': round(dim2, 3)
                }
            else:  # 长条形
                result['shape'] = '长条形'
                result['dimensions'] = {
                    '长度': round(dim1, 3),
                    '宽度': round(dim2, 3)
                }
            
            return result
            
        except Exception as e:
            logger.warning(f"分析面形状时出错: {str(e)}")
            return {'shape': '未知', 'dimensions': {}}
    
    @staticmethod
    def _estimate_opening_dimensions(face, adjacent_faces, geometry) -> Dict:
        """估算开口尺寸"""
        try:
            # 简化估算：基于底面尺寸和锥度
            bottom_bbox = PocketAnalyzer._get_face_bounding_box(face)
            if not bottom_bbox:
                return {}
            
            taper = geometry.get('taper', 0) if geometry else 0
            depth = geometry.get('depth', 0) if geometry else 0
            
            if taper > 0 and depth > 0:
                # 有锥度的情况
                taper_rad = math.radians(taper)
                expansion = depth * math.tan(taper_rad)
                
                return {
                    '长度': round(bottom_bbox['dx'] + 2 * expansion, 3),
                    '宽度': round(bottom_bbox['dy'] + 2 * expansion, 3)
                }
            else:
                # 直壁的情况
                return {
                    '长度': round(bottom_bbox['dx'], 3),
                    '宽度': round(bottom_bbox['dy'], 3)
                }
            
        except Exception as e:
            logger.warning(f"估算开口尺寸时出错: {str(e)}")
            return {}
    
    @staticmethod
    def _estimate_pocket_volume(face, adjacent_faces, geometry) -> float:
        """估算凹槽体积"""
        try:
            if not geometry:
                return 0.0
            
            depth = geometry.get('depth', 0)
            bottom_area = geometry.get('bottom_area', 0)
            opening_area = geometry.get('opening_area', 0)
            
            if depth > 0 and bottom_area > 0 and opening_area > 0:
                # 使用截锥体公式：V = h/3 * (A1 + A2 + sqrt(A1*A2))
                volume = depth / 3 * (bottom_area + opening_area + math.sqrt(bottom_area * opening_area))
                return volume
            elif depth > 0 and bottom_area > 0:
                # 简化计算：柱体体积
                return bottom_area * depth
            
            return 0.0
            
        except Exception as e:
            logger.warning(f"估算凹槽体积时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _analyze_closure_type(face, adjacent_faces) -> str:
        """分析封闭类型"""
        try:
            sidewall_count = len(adjacent_faces)
            
            if sidewall_count >= 4:
                return "封闭凹槽"
            elif sidewall_count == 3:
                return "三面封闭"
            elif sidewall_count == 2:
                return "半封闭凹槽"
            else:
                return "开放凹槽"
                
        except Exception as e:
            logger.warning(f"分析封闭类型时出错: {str(e)}")
            return "未知"
    
    @staticmethod
    def _analyze_pocket_purpose(face, adjacent_faces, geometry) -> str:
        """分析凹槽的可能用途"""
        try:
            if not geometry:
                return "未知"
            
            depth = geometry.get('depth', 0)
            bottom_area = geometry.get('bottom_area', 0)
            opening_area = geometry.get('opening_area', 0)
            
            # 基于尺寸特征判断用途
            if bottom_area > 0 and opening_area > 0:
                area_ratio = bottom_area / opening_area
                
                if depth > 0:
                    aspect_ratio = depth / math.sqrt(bottom_area)
                    
                    if aspect_ratio > 1.5:
                        return "深槽/容器"
                    elif aspect_ratio > 0.5:
                        if area_ratio > 0.8:
                            return "定位槽"
                        else:
                            return "锥形槽"
                    elif aspect_ratio < 0.2:
                        return "浅槽/减重"
                    else:
                        return "通用凹槽"
            
            return "通用凹槽"
            
        except Exception as e:
            logger.warning(f"分析凹槽用途时出错: {str(e)}")
            return "未知"
    
    @staticmethod
    def _get_face_bounding_box(face) -> Optional[Dict]:
        """获取面的边界框"""
        try:

            bbox = Bnd_Box()
            brepbndlib.Add(face, bbox)
            
            xmin, ymin, zmin, xmax, ymax, zmax = bbox.Get()
            
            return {
                'xmin': xmin, 'ymin': ymin, 'zmin': zmin,
                'xmax': xmax, 'ymax': ymax, 'zmax': zmax,
                'dx': xmax - xmin,
                'dy': ymax - ymin,
                'dz': zmax - zmin
            }
            
        except Exception as e:
            logger.warning(f"获取边界框时出错: {str(e)}")
            return None
    
    @staticmethod
    def _calculate_surface_area(face) -> float:
        """计算面的表面积"""
        try:
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            return props.Mass()
        except Exception as e:
            logger.warning(f"计算面积时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _get_face_center(face) -> Optional[gp_Pnt]:
        """获取面的中心点"""
        try:
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            return props.CentreOfMass()
        except Exception as e:
            logger.warning(f"获取面中心时出错: {str(e)}")
            return None