#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
孔特征分析模块 - 增强版
改进内容：
1. 多策略综合判断孔识别
2. 从边界圆提取实际孔位置和深度
3. 边界连接分析判断通孔/盲孔
4. 改进交叉孔检测
"""

import logging
import math
import numpy as np
from typing import Dict, List, Tuple, Optional, Any

from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import (
    GeomAbs_Cylinder, GeomAbs_Circle, GeomAbs_BSplineCurve, 
    GeomAbs_Line, GeomAbs_Cone, GeomAbs_Plane
)
from OCC.Core.gp import gp_Pnt, gp_Vec, gp_Dir
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.TopAbs import TopAbs_EDGE, TopAbs_WIRE, TopAbs_FACE
from OCC.Core.TopoDS import topods
from OCC.Core.BRepClass3d import BRepClass3d_SolidClassifier
from OCC.Core.TopAbs import TopAbs_IN, TopAbs_OUT, TopAbs_ON
from OCC.Core.Precision import precision
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop

logger = logging.getLogger(__name__)


class HoleAnalyzer:
    """孔特征分析类 - 增强版"""
    
    # 配置参数
    CAVITY_THRESHOLD = 0.6  # 空腔比例阈值
    U_RANGE_THRESHOLD = 1.5 * np.pi  # U参数域阈值
    DEPTH_SEARCH_STEP = 0.1  # 深度搜索步长(mm)
    
    @staticmethod
    def is_hole(cylindrical_face, shape) -> bool:
        """
        判断一个圆柱面是否构成孔 - 多策略综合判断
        
        策略1: 多点采样检测空腔 (96个采样点)
        策略2: 轴向开口检测
        策略3: 参数域分析
        策略4: 边界特征检查
        
        Args:
            cylindrical_face: 圆柱面
            shape: 整体形状
            
        Returns:
            bool: 是否为孔
        """
        try:
            # 检查圆柱面类型
            surf = BRepAdaptor_Surface(cylindrical_face)
            if surf.GetType() != GeomAbs_Cylinder:
                return False
            
            cylinder = surf.Cylinder()
            radius = cylinder.Radius()
            axis = cylinder.Axis()
            location = cylinder.Location()
            direction = axis.Direction()
            
            # 获取轴向方向向量
            axis_dir = gp_Vec(direction.X(), direction.Y(), direction.Z())
            mag = axis_dir.Magnitude()
            if mag < 1e-10:
                return False
            axis_dir.Divide(mag)
            
            center_point = gp_Pnt(location.X(), location.Y(), location.Z())
            
            # 创建垂直于轴向的两个正交向量
            perp_vec1 = HoleAnalyzer._create_perpendicular_vector(axis_dir)
            perp_vec2 = axis_dir.Crossed(perp_vec1)
            if perp_vec2.Magnitude() > 1e-10:
                perp_vec2.Divide(perp_vec2.Magnitude())
            
            # ===== 策略1: 多点采样检测空腔 =====
            inside_count = 0
            outside_count = 0
            
            # 在不同半径、角度、轴向位置采样 (4 × 8 × 3 = 96个点)
            radius_ratios = [0.2, 0.4, 0.6, 0.8]
            angles = np.linspace(0, 2 * np.pi, 8, endpoint=False)
            axial_offsets = [-radius * 0.5, 0, radius * 0.5]
            
            for r_ratio in radius_ratios:
                r = radius * r_ratio
                for angle in angles:
                    for axial_off in axial_offsets:
                        x_offset = r * np.cos(angle)
                        y_offset = r * np.sin(angle)
                        
                        sample_point = gp_Pnt(
                            center_point.X() + perp_vec1.X() * x_offset + perp_vec2.X() * y_offset + axis_dir.X() * axial_off,
                            center_point.Y() + perp_vec1.Y() * x_offset + perp_vec2.Y() * y_offset + axis_dir.Y() * axial_off,
                            center_point.Z() + perp_vec1.Z() * x_offset + perp_vec2.Z() * y_offset + axis_dir.Z() * axial_off
                        )
                        
                        if HoleAnalyzer._check_point_in_shape(sample_point, shape):
                            inside_count += 1
                        else:
                            outside_count += 1
            
            total_samples = inside_count + outside_count
            if total_samples == 0:
                return False
            
            outside_ratio = outside_count / total_samples
            
            # 如果超过60%的内部点在形状外部（空腔），认为是孔
            if outside_ratio > HoleAnalyzer.CAVITY_THRESHOLD:
                logger.debug(f"策略1识别为孔：空腔比例 {outside_ratio:.2%}")
                return True
            
            # ===== 策略2: 检查轴向开口 =====
            if HoleAnalyzer._check_axial_opening(center_point, axis_dir, radius, shape, perp_vec1, perp_vec2):
                logger.debug("策略2：通过轴向开口检测识别为孔")
                return True
            
            # ===== 策略3: 检查圆柱面参数域 =====
            u_min = surf.FirstUParameter()
            u_max = surf.LastUParameter()
            u_range = u_max - u_min
            
            # 孔的圆柱面U方向应该接近完整圆周(2π)或至少半圆以上
            if u_range > HoleAnalyzer.U_RANGE_THRESHOLD and outside_ratio > 0.3:
                logger.debug(f"策略3识别为孔：U范围={u_range:.3f}, 空腔比例={outside_ratio:.2%}")
                return True
            
            # ===== 策略4: 边界特征检查 =====
            if HoleAnalyzer._check_hole_boundary_features(cylindrical_face):
                if outside_ratio > 0.25:
                    logger.debug("策略4：通过边界特征识别为孔")
                    return True
            
            logger.debug(f"不是孔：空腔比例={outside_ratio:.2%}")
            return False
            
        except Exception as e:
            logger.warning(f"孔特征判断时出错: {str(e)}")
            return False
    
    @staticmethod
    def analyze_hole(cylindrical_face, shape) -> Dict[str, Any]:
        """
        分析孔的属性 - 增强版
        
        Args:
            cylindrical_face: 孔的圆柱面
            shape: 整体形状
            
        Returns:
            dict: 孔的属性
        """
        properties = {
            '类型': '孔',
            '直径': 0.0,
            '半径': 0.0,
            '深度': 0.0,
            '位置': None,
            '方向': None,
            '孔类型': '未知',
            '是否交叉孔': False,
            '边界信息': {}
        }
        
        try:
            surf = BRepAdaptor_Surface(cylindrical_face)
            if surf.GetType() != GeomAbs_Cylinder:
                return properties
            
            cylinder = surf.Cylinder()
            radius = cylinder.Radius()
            axis = cylinder.Axis()
            direction = axis.Direction()
            
            properties['半径'] = round(radius, 3)
            properties['直径'] = round(radius * 2, 3)
            properties['方向'] = (round(direction.X(), 6), round(direction.Y(), 6), round(direction.Z(), 6))
            
            # ===== 关键改进：从边界圆提取实际位置和深度 =====
            position, depth, boundary_info = HoleAnalyzer._get_hole_position_and_depth_from_bounds(
                cylindrical_face, shape, surf
            )
            
            if position:
                properties['位置'] = (round(position[0], 3), round(position[1], 3), round(position[2], 3))
            
            if depth > 0:
                properties['深度'] = round(depth, 3)
            else:
                # 备用方法：使用原始深度计算
                properties['深度'] = round(HoleAnalyzer.calculate_hole_depth(cylindrical_face, shape), 3)
            
            properties['边界信息'] = boundary_info
            
            # ===== 改进的孔类型判断 =====
            hole_type, is_intersecting = HoleAnalyzer._determine_hole_type(
                cylindrical_face, shape, boundary_info
            )
            properties['孔类型'] = hole_type
            properties['是否交叉孔'] = is_intersecting
            
        except Exception as e:
            logger.error(f"分析孔特征时出错: {str(e)}")
            properties['错误'] = str(e)
        
        return properties
    
    @staticmethod
    def _get_hole_position_and_depth_from_bounds(cylindrical_face, shape, surf) -> Tuple[Optional[Tuple], float, Dict]:
        """
        从圆柱面边界圆提取实际孔位置和深度 - 核心改进
        
        解决问题：cylinder.Location()返回的是STEP文件中的轴线参考点，
        不是孔的实际几何位置
        
        方法：
        1. 提取圆柱面的所有圆形边界
        2. 计算两个边界圆的中点作为孔中心
        3. 两圆心距离作为孔深度
        
        Returns:
            Tuple[position, depth, boundary_info]
        """
        boundary_info = {
            'boundary_count': 0,
            'boundary_positions': [],
            'boundary_types': [],
            'connected_faces': []
        }
        
        try:
            cylinder = surf.Cylinder()
            axis = cylinder.Axis()
            direction = axis.Direction()
            axis_dir = np.array([direction.X(), direction.Y(), direction.Z()])
            
            # 方法1: 从边界圆提取位置
            boundary_positions = []
            boundary_types = []
            
            explorer = TopExp_Explorer(cylindrical_face, TopAbs_EDGE)
            while explorer.More():
                edge = topods.Edge(explorer.Current())
                curve = BRepAdaptor_Curve(edge)
                curve_type = curve.GetType()
                
                if curve_type == GeomAbs_Circle:
                    circle = curve.Circle()
                    center = circle.Location()
                    boundary_positions.append(np.array([center.X(), center.Y(), center.Z()]))
                    
                    # 检查是完整圆还是部分圆弧
                    first_param = curve.FirstParameter()
                    last_param = curve.LastParameter()
                    param_range = abs(last_param - first_param)
                    
                    if param_range > 1.9 * np.pi:
                        boundary_types.append('完整圆')
                    else:
                        boundary_types.append('部分圆弧')
                
                explorer.Next()
            
            boundary_info['boundary_count'] = len(boundary_positions)
            boundary_info['boundary_types'] = boundary_types
            
            # 如果有两个边界圆，计算实际位置和深度
            if len(boundary_positions) >= 2:
                # 按轴向位置排序
                sorted_positions = sorted(boundary_positions, 
                                         key=lambda p: np.dot(p, axis_dir))
                
                p1 = sorted_positions[0]
                p2 = sorted_positions[-1]
                
                # 孔中心：两个边界圆中点
                center = (p1 + p2) / 2
                
                # 孔深度：两圆心距离
                depth = np.linalg.norm(p2 - p1)
                
                boundary_info['boundary_positions'] = [p1.tolist(), p2.tolist()]
                
                return tuple(center), depth, boundary_info
            
            # 方法2: 使用V参数域计算深度
            v_min = surf.FirstVParameter()
            v_max = surf.LastVParameter()
            
            if abs(v_max - v_min) > 1e-6:
                depth_from_params = abs(v_max - v_min)
                
                # 使用参数域中点作为位置（比Location更可靠）
                u_mid = (surf.FirstUParameter() + surf.LastUParameter()) / 2
                v_mid = (v_min + v_max) / 2
                
                center_point = surf.Value(u_mid, v_mid)
                position = (center_point.X(), center_point.Y(), center_point.Z())
                
                return position, depth_from_params, boundary_info
            
            return None, 0.0, boundary_info
            
        except Exception as e:
            logger.warning(f"从边界提取位置时出错: {str(e)}")
            return None, 0.0, boundary_info
    
    @staticmethod
    def _determine_hole_type(cylindrical_face, shape, boundary_info: Dict) -> Tuple[str, bool]:
        """
        确定孔类型（通孔/盲孔）- 边界连接分析
        
        改进：检查圆形边界连接的面类型
        - 连接锥面 → 封闭端（倒角或沉头）
        - 连接平面 → 封闭端（盲孔底部）
        - 无连接/空 → 开放端
        
        Returns:
            Tuple[孔类型, 是否交叉孔]
        """
        try:
            # 检查是否为交叉孔
            is_intersecting = HoleAnalyzer._check_if_intersecting_hole(cylindrical_face, shape)
            
            if is_intersecting:
                return '交叉孔', True
            
            # 分析边界连接
            open_ends = 0
            closed_ends = 0
            end_details = []
            
            explorer = TopExp_Explorer(cylindrical_face, TopAbs_EDGE)
            circular_edges = []
            
            while explorer.More():
                edge = topods.Edge(explorer.Current())
                curve = BRepAdaptor_Curve(edge)
                
                if curve.GetType() == GeomAbs_Circle:
                    # 检查是否是完整圆
                    first_param = curve.FirstParameter()
                    last_param = curve.LastParameter()
                    if abs(last_param - first_param) > 1.9 * np.pi:
                        circular_edges.append(edge)
                
                explorer.Next()
            
            # 对每个圆形边界，检查连接的面类型
            for edge in circular_edges:
                connected_type = HoleAnalyzer._get_connected_face_type(edge, cylindrical_face, shape)
                end_details.append(connected_type)
                
                if connected_type in ['cone', 'plane', 'sphere']:
                    closed_ends += 1
                elif connected_type == 'open':
                    open_ends += 1
            
            # 判断孔类型
            if closed_ends == 2:
                return '封闭腔', False
            elif closed_ends == 1 and open_ends >= 1:
                return '盲孔', False
            elif open_ends >= 2:
                # 进一步验证是否真的是通孔
                if HoleAnalyzer._verify_through_hole(cylindrical_face, shape):
                    return '通孔', False
                else:
                    return '盲孔', False
            elif closed_ends == 0 and open_ends == 0:
                # 可能是交叉孔或复杂结构
                return '未知', is_intersecting
            else:
                return '盲孔', False
            
        except Exception as e:
            logger.warning(f"确定孔类型时出错: {str(e)}")
            return '未知', False
    
    @staticmethod
    def _get_connected_face_type(edge, cylindrical_face, shape) -> str:
        """
        获取边连接的面的类型
        
        Returns:
            'cone' - 锥面（倒角）
            'plane' - 平面（盲孔底）
            'cylinder' - 圆柱面（同一个孔或台阶孔）
            'sphere' - 球面
            'open' - 无连接（开放端）
        """
        try:
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while face_explorer.More():
                face = topods.Face(face_explorer.Current())
                
                # 跳过当前圆柱面
                if face.IsSame(cylindrical_face):
                    face_explorer.Next()
                    continue
                
                # 检查该面是否包含这条边
                edge_explorer = TopExp_Explorer(face, TopAbs_EDGE)
                while edge_explorer.More():
                    other_edge = topods.Edge(edge_explorer.Current())
                    
                    if edge.IsSame(other_edge):
                        # 找到连接面，判断类型
                        surf = BRepAdaptor_Surface(face)
                        surf_type = surf.GetType()
                        
                        if surf_type == GeomAbs_Cone:
                            return 'cone'
                        elif surf_type == GeomAbs_Plane:
                            return 'plane'
                        elif surf_type == GeomAbs_Cylinder:
                            return 'cylinder'
                        else:
                            return 'other'
                    
                    edge_explorer.Next()
                
                face_explorer.Next()
            
            return 'open'
            
        except Exception as e:
            logger.warning(f"获取连接面类型时出错: {str(e)}")
            return 'unknown'
    
    @staticmethod
    def _verify_through_hole(cylindrical_face, shape) -> bool:
        """
        验证是否真的是通孔 - 增强版
        检查两端是否都能到达形状外部
        """
        try:
            surf = BRepAdaptor_Surface(cylindrical_face)
            cylinder = surf.Cylinder()
            axis = cylinder.Axis()
            location = cylinder.Location()
            direction = axis.Direction()
            radius = cylinder.Radius()
            
            axis_dir = gp_Vec(direction.X(), direction.Y(), direction.Z())
            mag = axis_dir.Magnitude()
            if mag < 1e-10:
                return False
            axis_dir.Divide(mag)
            
            # 使用参数域中点作为起始点（比Location更可靠）
            v_mid = (surf.FirstVParameter() + surf.LastVParameter()) / 2
            u_mid = (surf.FirstUParameter() + surf.LastUParameter()) / 2
            center_point = surf.Value(u_mid, v_mid)
            
            perp_vec1 = HoleAnalyzer._create_perpendicular_vector(axis_dir)
            perp_vec2 = axis_dir.Crossed(perp_vec1)
            if perp_vec2.Magnitude() > 1e-10:
                perp_vec2.Divide(perp_vec2.Magnitude())
            
            check_distances = [radius * 2, radius * 5, radius * 10, 50, 100, 200]
            
            # 正向检查
            forward_open = False
            for dist in check_distances:
                test_point = gp_Pnt(
                    center_point.X() + axis_dir.X() * dist,
                    center_point.Y() + axis_dir.Y() * dist,
                    center_point.Z() + axis_dir.Z() * dist
                )
                if not HoleAnalyzer._check_point_in_shape(test_point, shape):
                    if HoleAnalyzer._verify_opening(center_point, axis_dir, dist, radius, shape, perp_vec1, perp_vec2):
                        forward_open = True
                        break
            
            # 负向检查
            backward_open = False
            for dist in check_distances:
                test_point = gp_Pnt(
                    center_point.X() - axis_dir.X() * dist,
                    center_point.Y() - axis_dir.Y() * dist,
                    center_point.Z() - axis_dir.Z() * dist
                )
                if not HoleAnalyzer._check_point_in_shape(test_point, shape):
                    if HoleAnalyzer._verify_opening(center_point, axis_dir, -dist, radius, shape, perp_vec1, perp_vec2):
                        backward_open = True
                        break
            
            return forward_open and backward_open
            
        except Exception as e:
            logger.error(f"验证通孔时出错: {str(e)}")
            return False
    
    @staticmethod
    def calculate_hole_depth(cylindrical_face, shape) -> float:
        """
        计算孔的深度 - 双阶段搜索
        
        改进：
        1. 粗搜索快速定位边界区域
        2. 细搜索精确定位边界位置
        3. 动态步长适应不同孔径
        """
        try:
            surf = BRepAdaptor_Surface(cylindrical_face)
            cylinder = surf.Cylinder()
            radius = cylinder.Radius()
            axis = cylinder.Axis()
            direction = axis.Direction()
            
            axis_dir = gp_Vec(direction.X(), direction.Y(), direction.Z())
            mag = axis_dir.Magnitude()
            if mag < 1e-10:
                return 0.0
            axis_dir.Divide(mag)
            
            # 使用参数域中点作为起始点
            v_mid = (surf.FirstVParameter() + surf.LastVParameter()) / 2
            u_mid = (surf.FirstUParameter() + surf.LastUParameter()) / 2
            center_point = surf.Value(u_mid, v_mid)
            
            perp_vec1 = HoleAnalyzer._create_perpendicular_vector(axis_dir)
            perp_vec2 = axis_dir.Crossed(perp_vec1)
            if perp_vec2.Magnitude() > 1e-10:
                perp_vec2.Divide(perp_vec2.Magnitude())
            
            # 动态步长：根据孔径调整
            coarse_step = max(radius * 0.5, 1.0)  # 粗搜索步长
            fine_step = HoleAnalyzer.DEPTH_SEARCH_STEP  # 细搜索步长 0.1mm
            max_search_distance = 500.0
            
            # 在两个方向搜索边界
            forward_depth = HoleAnalyzer._search_boundary(
                center_point, axis_dir, radius, shape, 
                perp_vec1, perp_vec2, coarse_step, fine_step, max_search_distance
            )
            
            neg_axis_dir = gp_Vec(-axis_dir.X(), -axis_dir.Y(), -axis_dir.Z())
            backward_depth = HoleAnalyzer._search_boundary(
                center_point, neg_axis_dir, radius, shape,
                perp_vec1, perp_vec2, coarse_step, fine_step, max_search_distance
            )
            
            total_depth = forward_depth + backward_depth
            
            logger.debug(f"孔深度计算: forward={forward_depth:.3f}, backward={backward_depth:.3f}, total={total_depth:.3f}")
            
            return total_depth
            
        except Exception as e:
            logger.error(f"计算孔深度时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _search_boundary(center, axis_dir, radius, shape, perp1, perp2, 
                         coarse_step, fine_step, max_distance) -> float:
        """
        双阶段搜索边界位置
        """
        # 阶段1: 粗搜索
        coarse_boundary = 0.0
        dist = coarse_step
        
        while dist < max_distance:
            test_point = gp_Pnt(
                center.X() + axis_dir.X() * dist,
                center.Y() + axis_dir.Y() * dist,
                center.Z() + axis_dir.Z() * dist
            )
            
            # 检查该位置圆周上的点
            in_cavity = HoleAnalyzer._check_cavity_at_position(
                test_point, axis_dir, radius * 0.5, perp1, perp2, shape
            )
            
            if not in_cavity:
                coarse_boundary = dist
                break
            
            dist += coarse_step
        
        if coarse_boundary == 0.0:
            return 0.0
        
        # 阶段2: 细搜索（在粗边界附近）
        fine_start = max(0, coarse_boundary - coarse_step)
        fine_end = coarse_boundary + fine_step
        
        dist = fine_start
        while dist < fine_end:
            test_point = gp_Pnt(
                center.X() + axis_dir.X() * dist,
                center.Y() + axis_dir.Y() * dist,
                center.Z() + axis_dir.Z() * dist
            )
            
            in_cavity = HoleAnalyzer._check_cavity_at_position(
                test_point, axis_dir, radius * 0.5, perp1, perp2, shape
            )
            
            if not in_cavity:
                return dist
            
            dist += fine_step
        
        return coarse_boundary
    
    @staticmethod
    def _check_cavity_at_position(center, axis_dir, check_radius, perp1, perp2, shape) -> bool:
        """
        检查某个轴向位置是否是空腔
        """
        empty_count = 0
        angles = np.linspace(0, 2 * np.pi, 8, endpoint=False)
        
        for angle in angles:
            x_off = check_radius * np.cos(angle)
            y_off = check_radius * np.sin(angle)
            
            sample_point = gp_Pnt(
                center.X() + perp1.X() * x_off + perp2.X() * y_off,
                center.Y() + perp1.Y() * x_off + perp2.Y() * y_off,
                center.Z() + perp1.Z() * x_off + perp2.Z() * y_off
            )
            
            if not HoleAnalyzer._check_point_in_shape(sample_point, shape):
                empty_count += 1
        
        # 如果超过一半的点在形状外，认为是空腔
        return empty_count >= 4
    
    @staticmethod
    def _check_axial_opening(center_point, axis_dir, radius, shape, perp_vec1, perp_vec2) -> bool:
        """
        检查沿轴向是否有开口
        """
        check_distances = [radius * 2, radius * 5, radius * 10, radius * 20, 50.0, 100.0]
        
        for direction in [1, -1]:
            for dist in check_distances:
                axial_center = gp_Pnt(
                    center_point.X() + axis_dir.X() * dist * direction,
                    center_point.Y() + axis_dir.Y() * dist * direction,
                    center_point.Z() + axis_dir.Z() * dist * direction
                )
                
                if not HoleAnalyzer._check_point_in_shape(axial_center, shape):
                    empty_count = 0
                    sample_angles = np.linspace(0, 2 * np.pi, 8, endpoint=False)
                    
                    for angle in sample_angles:
                        x_off = radius * 0.5 * np.cos(angle)
                        y_off = radius * 0.5 * np.sin(angle)
                        
                        sample_point = gp_Pnt(
                            axial_center.X() + perp_vec1.X() * x_off + perp_vec2.X() * y_off,
                            axial_center.Y() + perp_vec1.Y() * x_off + perp_vec2.Y() * y_off,
                            axial_center.Z() + perp_vec1.Z() * x_off + perp_vec2.Z() * y_off
                        )
                        
                        if not HoleAnalyzer._check_point_in_shape(sample_point, shape):
                            empty_count += 1
                    
                    if empty_count >= 4:
                        return True
                    break
        
        return False
    
    @staticmethod
    def _check_hole_boundary_features(cylindrical_face) -> bool:
        """
        检查圆柱面的边界特征是否符合孔的特点
        """
        try:
            explorer = TopExp_Explorer(cylindrical_face, TopAbs_EDGE)
            
            circular_count = 0
            complete_circle_count = 0
            
            while explorer.More():
                edge = topods.Edge(explorer.Current())
                curve = BRepAdaptor_Curve(edge)
                
                if curve.GetType() == GeomAbs_Circle:
                    circular_count += 1
                    
                    first_param = curve.FirstParameter()
                    last_param = curve.LastParameter()
                    if abs(last_param - first_param) > 1.9 * np.pi:
                        complete_circle_count += 1
                
                explorer.Next()
            
            # 孔通常有1-2个完整的圆形边界
            return complete_circle_count >= 1 or circular_count >= 2
            
        except Exception as e:
            logger.warning(f"检查边界特征时出错: {str(e)}")
            return False
    
    @staticmethod
    def _check_if_intersecting_hole(cylindrical_face, shape) -> bool:
        """
        检查是否是交叉孔
        """
        try:
            explorer = TopExp_Explorer(cylindrical_face, TopAbs_EDGE)
            
            edge_count = 0
            partial_circle_count = 0
            non_circular_count = 0
            
            while explorer.More():
                edge = topods.Edge(explorer.Current())
                edge_count += 1
                
                curve = BRepAdaptor_Curve(edge)
                curve_type = curve.GetType()
                
                if curve_type == GeomAbs_Circle:
                    first_param = curve.FirstParameter()
                    last_param = curve.LastParameter()
                    param_range = abs(last_param - first_param)
                    
                    if param_range < 1.9 * np.pi:
                        partial_circle_count += 1
                        
                elif curve_type in [GeomAbs_Line, GeomAbs_BSplineCurve]:
                    non_circular_count += 1
                
                explorer.Next()
            
            # 交叉孔判断条件
            if partial_circle_count > 0:
                return True
            
            if edge_count > 4 and non_circular_count > 0:
                return True
            
            # 检查参数域完整性
            surf = BRepAdaptor_Surface(cylindrical_face)
            u_range = surf.LastUParameter() - surf.FirstUParameter()
            
            if u_range < 1.8 * np.pi:
                return True
            
            return False
            
        except Exception as e:
            logger.warning(f"检查交叉孔时出错: {str(e)}")
            return False
    
    @staticmethod
    def _verify_opening(center, axis_dir, axial_dist, radius, shape, perp1, perp2) -> bool:
        """
        验证某个轴向位置是否确实是开口
        """
        axial_center = gp_Pnt(
            center.X() + axis_dir.X() * axial_dist,
            center.Y() + axis_dir.Y() * axial_dist,
            center.Z() + axis_dir.Z() * axial_dist
        )
        
        empty_count = 0
        for angle in np.linspace(0, 2 * np.pi, 8, endpoint=False):
            x_off = radius * 0.5 * np.cos(angle)
            y_off = radius * 0.5 * np.sin(angle)
            
            sample_point = gp_Pnt(
                axial_center.X() + perp1.X() * x_off + perp2.X() * y_off,
                axial_center.Y() + perp1.Y() * x_off + perp2.Y() * y_off,
                axial_center.Z() + perp1.Z() * x_off + perp2.Z() * y_off
            )
            
            if not HoleAnalyzer._check_point_in_shape(sample_point, shape):
                empty_count += 1
        
        return empty_count >= 4
    
    @staticmethod
    def _create_perpendicular_vector(axis_dir: gp_Vec) -> gp_Vec:
        """
        创建一个垂直于给定向量的向量
        """
        # 选择一个不平行的参考向量
        if abs(axis_dir.X()) < 0.9:
            ref = gp_Vec(1, 0, 0)
        else:
            ref = gp_Vec(0, 1, 0)
        
        perp = axis_dir.Crossed(ref)
        mag = perp.Magnitude()
        if mag > 1e-10:
            perp.Divide(mag)
        
        return perp
    
    @staticmethod
    def _check_point_in_shape(point: gp_Pnt, shape) -> bool:
        """
        检查点是否在形状内部
        """
        try:
            classifier = BRepClass3d_SolidClassifier()
            classifier.Load(shape)
            classifier.Perform(point, precision.Confusion())
            state = classifier.State()
            return state == TopAbs_IN
        except:
            return False
