#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
内六角特征分析模块 - 新增
识别内六角槽（如内六角螺钉的工具槽）

特征识别：
1. 6个等面积平面呈60°均匀分布
2. 中心有圆柱面（孔）
3. 入口可能有45°倒角
"""

import logging
import math
import numpy as np
from typing import Dict, List, Tuple, Optional, Any

from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import GeomAbs_Plane, GeomAbs_Cylinder, GeomAbs_Cone
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.TopAbs import TopAbs_FACE, TopAbs_EDGE
from OCC.Core.TopoDS import topods
from OCC.Core.gp import gp_Pnt, gp_Vec, gp_Dir
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop
from OCC.Core.BRepClass3d import BRepClass3d_SolidClassifier
from OCC.Core.TopAbs import TopAbs_IN, TopAbs_OUT
from OCC.Core.Precision import precision

logger = logging.getLogger(__name__)


# 标准内六角对边尺寸 (mm) - ISO标准
STANDARD_HEX_SIZES = {
    # 螺钉规格: 对边尺寸
    'M1.6': 1.5,
    'M2': 1.5,
    'M2.5': 2.0,
    'M3': 2.5,
    'M4': 3.0,
    'M5': 4.0,
    'M6': 5.0,
    'M8': 6.0,
    'M10': 8.0,
    'M12': 10.0,
    'M14': 12.0,
    'M16': 14.0,
    'M20': 17.0,
    'M24': 19.0,
}

# 对边尺寸到螺钉规格的反向映射
HEX_SIZE_TO_SCREW = {v: k for k, v in STANDARD_HEX_SIZES.items()}


class HexSocketAnalyzer:
    """内六角特征分析器"""
    
    # 配置参数
    ANGLE_TOLERANCE = 5.0  # 角度容差（度）
    AREA_TOLERANCE = 0.15  # 面积容差（比例）
    HEX_ANGLE = 60.0  # 六边形相邻面夹角
    
    @staticmethod
    def find_hex_socket_candidates(shape) -> List[Dict]:
        """
        在形状中查找可能的内六角特征
        
        方法：
        1. 找出所有平面
        2. 按面积分组
        3. 检查是否有6个相同面积的平面呈60°分布
        
        Args:
            shape: 整体形状
            
        Returns:
            List[Dict]: 候选内六角特征列表
        """
        candidates = []
        
        try:
            # 收集所有平面及其属性
            planes = []
            
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            while face_explorer.More():
                face = topods.Face(face_explorer.Current())
                surf = BRepAdaptor_Surface(face)
                
                if surf.GetType() == GeomAbs_Plane:
                    plane = surf.Plane()
                    normal = plane.Axis().Direction()
                    location = plane.Location()
                    
                    # 计算面积
                    props = GProp_GProps()
                    brepgprop.SurfaceProperties(face, props)
                    area = props.Mass()
                    center = props.CentreOfMass()
                    
                    planes.append({
                        'face': face,
                        'normal': (normal.X(), normal.Y(), normal.Z()),
                        'location': (location.X(), location.Y(), location.Z()),
                        'center': (center.X(), center.Y(), center.Z()),
                        'area': area
                    })
                
                face_explorer.Next()
            
            if len(planes) < 6:
                return candidates
            
            # 按面积分组（找相同面积的平面）
            area_groups = HexSocketAnalyzer._group_by_area(planes)
            
            # 检查每个有6个或更多成员的组
            for area, group in area_groups.items():
                if len(group) >= 6:
                    # 检查是否构成内六角
                    hex_result = HexSocketAnalyzer._check_hex_pattern(group, shape)
                    if hex_result:
                        candidates.append(hex_result)
            
            return candidates
            
        except Exception as e:
            logger.error(f"查找内六角候选时出错: {str(e)}")
            return candidates
    
    @staticmethod
    def _group_by_area(planes: List[Dict]) -> Dict[float, List[Dict]]:
        """
        按面积分组平面
        """
        groups = {}
        tolerance = 0.1  # 面积分组容差
        
        for plane in planes:
            area = plane['area']
            
            # 找到最接近的组
            found_group = None
            for group_area in groups.keys():
                if abs(area - group_area) / max(area, group_area) < HexSocketAnalyzer.AREA_TOLERANCE:
                    found_group = group_area
                    break
            
            if found_group is not None:
                groups[found_group].append(plane)
            else:
                groups[area] = [plane]
        
        return groups
    
    @staticmethod
    def _check_hex_pattern(planes: List[Dict], shape) -> Optional[Dict]:
        """
        检查一组平面是否构成内六角图案
        
        内六角特征：
        1. 6个平面的法向量两两夹角为60°（或120°）
        2. 中心有共同的轴线
        3. 是凹特征（向内）
        """
        if len(planes) < 6:
            return None
        
        try:
            # 计算所有平面中心的几何中心
            centers = [np.array(p['center']) for p in planes]
            geometric_center = np.mean(centers, axis=0)
            
            # 检查法向量分布
            normals = [np.array(p['normal']) for p in planes]
            
            # 统计法向量之间的角度
            angle_pairs = []
            for i in range(len(normals)):
                for j in range(i + 1, len(normals)):
                    dot = np.dot(normals[i], normals[j])
                    dot = np.clip(dot, -1, 1)
                    angle = np.degrees(np.arccos(abs(dot)))
                    angle_pairs.append((i, j, angle))
            
            # 内六角的法向量应该有特定的角度模式
            # 相邻面夹角60°，对面夹角0°（平行）
            angles_60 = [ap for ap in angle_pairs if 55 <= ap[2] <= 65]
            angles_0 = [ap for ap in angle_pairs if ap[2] <= 5]
            
            # 6个面应该有3对平行面，每对夹角0°
            # 相邻面夹角60°
            if len(angles_0) >= 3 and len(angles_60) >= 6:
                # 进一步验证：检查是否是凹特征
                is_recessed = HexSocketAnalyzer._check_if_recessed(planes, geometric_center, shape)
                
                if is_recessed:
                    # 计算对边距离
                    across_flats = HexSocketAnalyzer._calculate_across_flats(planes)
                    
                    # 匹配标准尺寸
                    standard_size = HexSocketAnalyzer._match_standard_size(across_flats)
                    
                    return {
                        '类型': '内六角槽',
                        '对边距离': round(across_flats, 3),
                        '标准规格': standard_size,
                        '中心位置': (round(geometric_center[0], 3), round(geometric_center[1], 3), round(geometric_center[2], 3)),
                        '面积': round(planes[0]['area'], 3),
                        '面数': len(planes),
                        '构成面': [p['face'] for p in planes[:6]]
                    }
            
            return None
            
        except Exception as e:
            logger.warning(f"检查内六角图案时出错: {str(e)}")
            return None
    
    @staticmethod
    def _check_if_recessed(planes: List[Dict], center: np.ndarray, shape) -> bool:
        """
        检查是否是凹特征（内六角槽应该是向内的）
        """
        try:
            # 检查中心点是否在形状外部（空腔）
            center_point = gp_Pnt(center[0], center[1], center[2])
            
            classifier = BRepClass3d_SolidClassifier()
            classifier.Load(shape)
            classifier.Perform(center_point, precision.Confusion())
            state = classifier.State()
            
            # 如果中心在形状外部，说明是凹特征
            return state == TopAbs_OUT
            
        except Exception as e:
            logger.warning(f"检查凹特征时出错: {str(e)}")
            return False
    
    @staticmethod
    def _calculate_across_flats(planes: List[Dict]) -> float:
        """
        计算内六角的对边距离
        
        方法：找到平行的对面，计算它们之间的距离
        """
        try:
            normals = [np.array(p['normal']) for p in planes]
            centers = [np.array(p['center']) for p in planes]
            
            # 找平行的面对
            parallel_pairs = []
            for i in range(len(normals)):
                for j in range(i + 1, len(normals)):
                    dot = abs(np.dot(normals[i], normals[j]))
                    if dot > 0.99:  # 几乎平行
                        # 计算两面中心的距离在法向量方向上的投影
                        center_diff = centers[j] - centers[i]
                        distance = abs(np.dot(center_diff, normals[i]))
                        parallel_pairs.append(distance)
            
            if parallel_pairs:
                # 返回平均对边距离
                return np.mean(parallel_pairs)
            
            return 0.0
            
        except Exception as e:
            logger.warning(f"计算对边距离时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _match_standard_size(across_flats: float) -> str:
        """
        匹配标准内六角尺寸
        """
        if across_flats <= 0:
            return '非标准'
        
        # 查找最接近的标准尺寸
        min_diff = float('inf')
        best_match = None
        
        for screw, size in STANDARD_HEX_SIZES.items():
            diff = abs(size - across_flats)
            if diff < min_diff:
                min_diff = diff
                best_match = screw
        
        # 如果差异在0.3mm以内，认为匹配
        if min_diff < 0.3 and best_match:
            return best_match
        
        return f'非标准({across_flats:.1f}mm)'
    
    @staticmethod
    def analyze_hex_socket(hex_faces: List, shape) -> Dict[str, Any]:
        """
        分析内六角槽的详细属性
        
        Args:
            hex_faces: 构成内六角的6个平面
            shape: 整体形状
            
        Returns:
            Dict: 内六角属性
        """
        properties = {
            '类型': '内六角槽',
            '对边距离': 0.0,
            '深度': 0.0,
            '入口倒角': None,
            '标准规格': '未知',
            '中心位置': None,
            '方向': None
        }
        
        try:
            if len(hex_faces) < 6:
                return properties
            
            # 收集面的属性
            planes_data = []
            for face in hex_faces:
                surf = BRepAdaptor_Surface(face)
                if surf.GetType() == GeomAbs_Plane:
                    plane = surf.Plane()
                    normal = plane.Axis().Direction()
                    
                    props = GProp_GProps()
                    brepgprop.SurfaceProperties(face, props)
                    area = props.Mass()
                    center = props.CentreOfMass()
                    
                    planes_data.append({
                        'face': face,
                        'normal': np.array([normal.X(), normal.Y(), normal.Z()]),
                        'center': np.array([center.X(), center.Y(), center.Z()]),
                        'area': area
                    })
            
            if len(planes_data) < 6:
                return properties
            
            # 计算中心
            centers = [p['center'] for p in planes_data]
            geometric_center = np.mean(centers, axis=0)
            properties['中心位置'] = (round(geometric_center[0], 3), round(geometric_center[1], 3), round(geometric_center[2], 3))
            
            # 计算对边距离
            across_flats = HexSocketAnalyzer._calculate_across_flats(planes_data)
            properties['对边距离'] = round(across_flats, 3)
            
            # 匹配标准
            properties['标准规格'] = HexSocketAnalyzer._match_standard_size(across_flats)
            
            # 计算深度（通过边界框）
            depth = HexSocketAnalyzer._calculate_hex_depth(planes_data)
            properties['深度'] = round(depth, 3)
            
            # 计算方向（六角槽的轴向）
            direction = HexSocketAnalyzer._calculate_hex_direction(planes_data)
            if direction is not None:
                properties['方向'] = (round(direction[0], 6), round(direction[1], 6), round(direction[2], 6))
            
            # 检查入口倒角
            chamfer = HexSocketAnalyzer._find_entrance_chamfer(hex_faces, shape)
            if chamfer:
                properties['入口倒角'] = chamfer
            
            return properties
            
        except Exception as e:
            logger.error(f"分析内六角槽时出错: {str(e)}")
            properties['错误'] = str(e)
            return properties
    
    @staticmethod
    def _calculate_hex_depth(planes_data: List[Dict]) -> float:
        """
        计算内六角深度
        """
        try:
            # 使用Z坐标范围估算深度
            z_coords = [p['center'][2] for p in planes_data]
            return max(z_coords) - min(z_coords)
        except:
            return 0.0
    
    @staticmethod
    def _calculate_hex_direction(planes_data: List[Dict]) -> Optional[np.ndarray]:
        """
        计算内六角的轴向方向
        
        方法：所有面的法向量叉积的平均方向
        """
        try:
            normals = [p['normal'] for p in planes_data]
            
            # 找到两个不平行的法向量
            n1 = normals[0]
            n2 = None
            
            for n in normals[1:]:
                dot = abs(np.dot(n1, n))
                if dot < 0.9:  # 不平行
                    n2 = n
                    break
            
            if n2 is not None:
                # 轴向是两个法向量的叉积
                axis = np.cross(n1, n2)
                mag = np.linalg.norm(axis)
                if mag > 1e-6:
                    return axis / mag
            
            return None
            
        except:
            return None
    
    @staticmethod
    def _find_entrance_chamfer(hex_faces: List, shape) -> Optional[Dict]:
        """
        查找内六角入口处的倒角
        """
        try:
            # 查找与内六角面相邻的锥面
            for hex_face in hex_faces:
                edge_explorer = TopExp_Explorer(hex_face, TopAbs_EDGE)
                
                while edge_explorer.More():
                    edge = topods.Edge(edge_explorer.Current())
                    
                    # 查找共享这条边的其他面
                    face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
                    while face_explorer.More():
                        other_face = topods.Face(face_explorer.Current())
                        
                        if other_face not in hex_faces:
                            surf = BRepAdaptor_Surface(other_face)
                            
                            if surf.GetType() == GeomAbs_Cone:
                                # 检查是否共享边
                                other_edge_explorer = TopExp_Explorer(other_face, TopAbs_EDGE)
                                while other_edge_explorer.More():
                                    other_edge = topods.Edge(other_edge_explorer.Current())
                                    if edge.IsSame(other_edge):
                                        cone = surf.Cone()
                                        angle = math.degrees(cone.SemiAngle())
                                        
                                        return {
                                            '角度': round(angle, 1),
                                            '类型': '锥形倒角'
                                        }
                                    other_edge_explorer.Next()
                        
                        face_explorer.Next()
                    
                    edge_explorer.Next()
            
            return None
            
        except Exception as e:
            logger.warning(f"查找入口倒角时出错: {str(e)}")
            return None
    
    @staticmethod
    def is_hex_socket_face(face, shape) -> bool:
        """
        快速判断一个面是否可能是内六角槽的一部分
        
        用于在特征识别流程中快速筛选
        """
        try:
            surf = BRepAdaptor_Surface(face)
            
            # 必须是平面
            if surf.GetType() != GeomAbs_Plane:
                return False
            
            # 面积应该较小（内六角槽的单个面）
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            area = props.Mass()
            
            # 典型内六角面积范围：1-200 mm²
            if area < 1 or area > 200:
                return False
            
            # 检查是否是凹特征
            center = props.CentreOfMass()
            
            # 沿法向量方向检查
            plane = surf.Plane()
            normal = plane.Axis().Direction()
            
            # 向内偏移检查点
            test_point = gp_Pnt(
                center.X() + normal.X() * 0.5,
                center.Y() + normal.Y() * 0.5,
                center.Z() + normal.Z() * 0.5
            )
            
            classifier = BRepClass3d_SolidClassifier()
            classifier.Load(shape)
            classifier.Perform(test_point, precision.Confusion())
            
            # 如果偏移点在形状外，说明是凹面
            return classifier.State() == TopAbs_OUT
            
        except Exception as e:
            logger.warning(f"检查内六角面时出错: {str(e)}")
            return False
