#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
改进的沟槽特征分析模块
增强对环形密封槽的识别能力
"""

import logging
import math
from typing import Dict, List, Optional
from OCC.Core.BRepBndLib import brepbndlib
from OCC.Core.TopAbs import TopAbs_FACE, TopAbs_EDGE
from OCC.Core.TopoDS import topods
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import (
    GeomAbs_Plane, GeomAbs_Cylinder, GeomAbs_Torus, 
    GeomAbs_BSplineSurface, GeomAbs_Line, GeomAbs_Circle
)
from OCC.Core.BRepBndLib import brepbndlib
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop
from OCC.Core.gp import gp_Pnt, gp_Vec, gp_Dir

logger = logging.getLogger(__name__)

class GrooveAnalyzer:
    """改进的沟槽特征分析器"""
    
    @staticmethod
    def is_groove(face, shape) -> bool:
        """
        判断一个面是否属于沟槽特征
        改进：增强对环形密封槽的识别
        """
        try:
            surf = BRepAdaptor_Surface(face)
            surf_type = surf.GetType()
            
            # 1. 环面是密封槽的典型特征！
            if surf_type == GeomAbs_Torus:
                # 环面通常就是O型圈槽
                torus = surf.Torus()
                major_radius = torus.MajorRadius()
                minor_radius = torus.MinorRadius()
                
                # 合理的密封槽尺寸比例
                if 0.1 < minor_radius / major_radius < 0.3:
                    logger.info(f"识别到环形密封槽: 大径={major_radius:.2f}, 小径={minor_radius:.2f}")
                    return True
            
            # 2. 平面或圆柱面底的沟槽
            elif surf_type in [GeomAbs_Plane, GeomAbs_Cylinder]:
                # 获取相邻面
                adjacent_faces = GrooveAnalyzer._get_adjacent_faces(face, shape)
                
                if len(adjacent_faces) < 2:
                    return False
                
                # 检查是否有环面相邻（密封槽的组合特征）
                has_torus_neighbor = False
                for adj_face in adjacent_faces:
                    adj_surf = BRepAdaptor_Surface(adj_face)
                    if adj_surf.GetType() == GeomAbs_Torus:
                        has_torus_neighbor = True
                        break
                
                if has_torus_neighbor:
                    logger.info("识别到与环面相邻的沟槽底面")
                    return True
                
                # 检查是否是凹陷结构
                if not GrooveAnalyzer._is_recessed_feature(face, shape):
                    return False
                
                # 分析几何特征
                geometry = GrooveAnalyzer._analyze_groove_geometry(face, adjacent_faces)
                
                if geometry:
                    length = geometry.get('length', 0)
                    width = geometry.get('width', 0)
                    depth = geometry.get('depth', 0)
                    
                    # 线性沟槽判断（长宽比大）
                    if length > 0 and width > 0 and depth > 0:
                        if length / width > 3.0 and 0.1 < depth / width < 1.0:
                            return True
                        
                        # 环形沟槽判断（长度接近周长）
                        if surf_type == GeomAbs_Cylinder:
                            cylinder = surf.Cylinder()
                            radius = cylinder.Radius()
                            circumference = 2 * math.pi * radius
                            
                            # 如果长度接近周长，可能是环形槽
                            if abs(length - circumference) / circumference < 0.2:
                                logger.info(f"识别到环形沟槽: 半径={radius:.2f}")
                                return True
            
            return False
            
        except Exception as e:
            logger.warning(f"沟槽特征判断时出错: {str(e)}")
            return False
    
    @staticmethod
    def analyze_groove(face, shape) -> Dict:
        """
        分析沟槽特征的属性
        改进：增加对环形密封槽的详细分析
        """
        try:
            properties = {
                '类型': '沟槽',
                '子类型': '未知',
                '长度': 0.0,
                '宽度': 0.0,
                '深度': 0.0,
                '底面类型': '未知',
                '特殊特征': []
            }
            
            surf = BRepAdaptor_Surface(face)
            surf_type = surf.GetType()
            
            # 环面密封槽分析
            if surf_type == GeomAbs_Torus:
                torus = surf.Torus()
                properties['子类型'] = 'O型圈槽'
                properties['底面类型'] = '环面'
                properties['大径'] = round(torus.MajorRadius() * 2, 3)
                properties['小径'] = round(torus.MinorRadius() * 2, 3)
                properties['槽深'] = round(torus.MinorRadius(), 3)
                properties['特殊特征'].append('密封槽')
                
                # 计算环形槽的"长度"（周长）
                properties['长度'] = round(2 * math.pi * torus.MajorRadius(), 3)
                properties['宽度'] = round(torus.MinorRadius() * 2, 3)
                properties['深度'] = round(torus.MinorRadius(), 3)
                
            else:
                # 分析其他类型的沟槽
                adjacent_faces = GrooveAnalyzer._get_adjacent_faces(face, shape)
                
                # 检查是否与环面相邻
                torus_count = 0
                for adj_face in adjacent_faces:
                    adj_surf = BRepAdaptor_Surface(adj_face)
                    if adj_surf.GetType() == GeomAbs_Torus:
                        torus_count += 1
                
                if torus_count > 0:
                    properties['子类型'] = '复合密封槽'
                    properties['特殊特征'].append(f'包含{torus_count}个环面特征')
                
                # 底面类型
                if surf_type == GeomAbs_Plane:
                    properties['底面类型'] = '平面'
                elif surf_type == GeomAbs_Cylinder:
                    properties['底面类型'] = '圆柱面'
                    cylinder = surf.Cylinder()
                    properties['圆柱半径'] = round(cylinder.Radius(), 3)
                
                # 几何尺寸分析
                geometry = GrooveAnalyzer._analyze_groove_geometry(face, adjacent_faces)
                if geometry:
                    properties['长度'] = round(geometry['length'], 3)
                    properties['宽度'] = round(geometry['width'], 3)
                    properties['深度'] = round(geometry['depth'], 3)
                    
                    # 判断沟槽类型
                    if properties['长度'] / properties['宽度'] > 5:
                        properties['子类型'] = '线性沟槽'
                    elif surf_type == GeomAbs_Cylinder:
                        circumference = 2 * math.pi * cylinder.Radius()
                        if abs(properties['长度'] - circumference) / circumference < 0.2:
                            properties['子类型'] = '环形沟槽'
            
            # 计算面积
            area = GrooveAnalyzer._calculate_surface_area(face)
            properties['底面积'] = round(area, 3)
            
            return properties
            
        except Exception as e:
            logger.error(f"分析沟槽特征时出错: {str(e)}")
            return {'类型': '沟槽', '错误': str(e)}
    
    @staticmethod
    def _get_adjacent_faces(face, shape) -> List:
        """获取与给定面相邻的面"""
        adjacent_faces = []
        
        try:
            # 获取面的所有边
            edge_explorer = TopExp_Explorer(face, TopAbs_EDGE)
            face_edges = []
            
            while edge_explorer.More():
                edge = topods.Edge(edge_explorer.Current())
                face_edges.append(edge)
                edge_explorer.Next()
            
            # 对每条边，找到共享这条边的其他面
            face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while face_explorer.More():
                other_face = topods.Face(face_explorer.Current())
                
                if not face.IsSame(other_face):
                    # 检查是否共享边
                    for edge in face_edges:
                        other_edge_explorer = TopExp_Explorer(other_face, TopAbs_EDGE)
                        
                        while other_edge_explorer.More():
                            other_edge = topods.Edge(other_edge_explorer.Current())
                            
                            if edge.IsSame(other_edge):
                                if other_face not in adjacent_faces:
                                    adjacent_faces.append(other_face)
                                break
                            
                            other_edge_explorer.Next()
                
                face_explorer.Next()
            
            return adjacent_faces
            
        except Exception as e:
            logger.warning(f"获取相邻面时出错: {str(e)}")
            return []
    
    @staticmethod
    def _is_recessed_feature(face, shape) -> bool:
        """判断是否是凹陷特征"""
        # 简化实现
        return True
    
    @staticmethod
    def _analyze_groove_geometry(face, adjacent_faces) -> Optional[Dict]:
        """分析沟槽的几何特征"""
        try:
            # 获取边界框来估算尺寸
            from OCC.Core.Bnd import Bnd_Box
            bbox = Bnd_Box()
            brepbndlib.Add(face, bbox)
            
            xmin, ymin, zmin, xmax, ymax, zmax = bbox.Get()
            
            dimensions = [xmax - xmin, ymax - ymin, zmax - zmin]
            dimensions.sort(reverse=True)
            
            # 估算深度
            depth = 0.0
            if adjacent_faces:
                # 通过相邻面估算深度
                for adj_face in adjacent_faces:
                    adj_surf = BRepAdaptor_Surface(adj_face)
                    if adj_surf.GetType() == GeomAbs_Torus:
                        # 如果相邻面是环面，深度就是环面的小半径
                        torus = adj_surf.Torus()
                        depth = max(depth, torus.MinorRadius())
            
            if depth == 0:
                depth = dimensions[2]  # 使用最小尺寸作为深度估算
            
            return {
                'length': dimensions[0],
                'width': dimensions[1],
                'depth': depth
            }
            
        except Exception as e:
            logger.warning(f"分析几何特征时出错: {str(e)}")
            return None
    
    @staticmethod
    def _calculate_surface_area(face) -> float:
        """计算面的表面积"""
        try:
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            return props.Mass()
        except Exception as e:
            logger.warning(f"计算面积时出错: {str(e)}")
            return 0.0