#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
几何分析模块：提供基本的几何分析功能，包括点、线、面的分析
"""

import logging
import math
import numpy as np
from OCC.Core.BRep import BRep_Tool
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import (
    GeomAbs_Plane, GeomAbs_Cylinder, GeomAbs_Cone, GeomAbs_Sphere,
    GeomAbs_Torus, GeomAbs_BezierSurface, GeomAbs_BSplineSurface,
    GeomAbs_SurfaceOfRevolution, GeomAbs_SurfaceOfExtrusion,
    GeomAbs_Line, GeomAbs_Circle, GeomAbs_Ellipse, GeomAbs_Hyperbola,
    GeomAbs_Parabola, GeomAbs_BezierCurve, GeomAbs_BSplineCurve,
    GeomAbs_OffsetCurve
)
from OCC.Core.TopAbs import TopAbs_OUT, TopAbs_IN, TopAbs_ON, TopAbs_UNKNOWN
from OCC.Core.BRepGProp import brepgprop  # Import the class instead of individual functions
from OCC.Core.GProp import GProp_GProps
from OCC.Core.gp import gp_Pnt, gp_Dir, gp_Vec
from OCC.Core.BRepClass3d import BRepClass3d_SolidClassifier
from OCC.Core.BRepBuilderAPI import BRepBuilderAPI_MakeVertex
from id2.feature_b import BSplineSurfaceParameterExtractor
from id2.feature_base import 特征类型
import warnings
 
warnings.filterwarnings("ignore", category=DeprecationWarning)
logger = logging.getLogger(__name__)

class GeometryAnalyzer:
    """几何分析类，提供基本的几何分析功能"""
    
    @staticmethod
    def analyze_surface(face):
        """
        分析面的表面类型
        
        Args:
            face: OCC面对象
            
        Returns:
            tuple: (表面类型, 属性字典)
        """
        surf = BRepAdaptor_Surface(face)
        surface_type = surf.GetType()
        
        properties = {
            '面积': GeometryAnalyzer.calculate_surface_area(face)
        }
        
        # 根据表面类型提取特定属性
        if surface_type == GeomAbs_Plane:
            plane = surf.Plane()
            loc = plane.Location()
            axis = plane.Axis()
            
            properties.update({
                '位置': (loc.X(), loc.Y(), loc.Z()),
                '方向': (axis.Direction().X(), axis.Direction().Y(), axis.Direction().Z())
            })
            return 特征类型['平面'], properties
        
        elif surface_type == GeomAbs_Cylinder:
            cylinder = surf.Cylinder()
            radius = cylinder.Radius()
            loc = cylinder.Location()
            axis = cylinder.Axis()
            
            properties.update({
                '半径': radius,
                '位置': (loc.X(), loc.Y(), loc.Z()),
                '方向': (axis.Direction().X(), axis.Direction().Y(), axis.Direction().Z())
            })
            return 特征类型['圆柱面'], properties
        
        elif surface_type == GeomAbs_Cone:
            cone = surf.Cone()
            radius = cone.RefRadius()
            semi_angle = cone.SemiAngle()
            loc = cone.Location()
            axis = cone.Axis()
            
            properties.update({
                '半径': radius,
                '半角': semi_angle,
                '位置': (loc.X(), loc.Y(), loc.Z()),
                '方向': (axis.Direction().X(), axis.Direction().Y(), axis.Direction().Z())
            })
            return 特征类型['圆锥面'], properties
        
        elif surface_type == GeomAbs_Sphere:
            sphere = surf.Sphere()
            radius = sphere.Radius()
            loc = sphere.Location()
            
            properties.update({
                '半径': radius,
                '位置': (loc.X(), loc.Y(), loc.Z())
            })
            return 特征类型['球面'], properties
        
        elif surface_type == GeomAbs_Torus:
            torus = surf.Torus()
            major_radius = torus.MajorRadius()
            minor_radius = torus.MinorRadius()
            loc = torus.Location()
            axis = torus.Axis()
            
            properties.update({
                '主半径': major_radius,
                '次半径': minor_radius,
                '位置': (loc.X(), loc.Y(), loc.Z()),
                '方向': (axis.Direction().X(), axis.Direction().Y(), axis.Direction().Z())
            })
            return 特征类型['环面'], properties
        
        elif surface_type == GeomAbs_BezierSurface:
            bezier = surf.Bezier()
            u_degree = bezier.UDegree()
            v_degree = bezier.VDegree()
            
            properties.update({
                'U度数': u_degree,
                'V度数': v_degree,
                'U极点数': bezier.NbUPoles(),
                'V极点数': bezier.NbVPoles()
            })
            return 特征类型['贝塞尔曲面'], properties
        
        elif surface_type == GeomAbs_BSplineSurface:
            bspline = surf.BSpline()
            u_degree = bspline.UDegree()
            v_degree = bspline.VDegree()
            params = BSplineSurfaceParameterExtractor.extract_parameters(surf)
    
            # 如果是B样条曲面，保存参数
            if params:
                 BSplineSurfaceParameterExtractor.save_to_json(params, "bspline_params.json")
                 properties.update(params)
            return 特征类型['B样条曲面'], properties
        
        elif surface_type == GeomAbs_SurfaceOfRevolution:
            # 旋转曲面属性
            return 特征类型['旋转面'], properties
        
        elif surface_type == GeomAbs_SurfaceOfExtrusion:
            # 拉伸曲面属性
            return 特征类型['拉伸面'], properties
        
        else:
            return 特征类型['未知'], properties
    
    @staticmethod
    def analyze_edge(edge):
        """
        分析边的类型
        
        Args:
            edge: OCC边对象
            
        Returns:
            dict: 边属性字典
        """
        curve_adaptor = BRepAdaptor_Curve(edge)
        curve_type = curve_adaptor.GetType()
        
        properties = {}
        
        # 根据曲线类型提取特定属性
        if curve_type == GeomAbs_Line:
            line = curve_adaptor.Line()
            direction = line.Direction()
            properties['类型'] = '直线'
            properties['方向'] = (direction.X(), direction.Y(), direction.Z())
        
        elif curve_type == GeomAbs_Circle:
            circle = curve_adaptor.Circle()
            radius = circle.Radius()
            loc = circle.Location()
            properties['类型'] = '圆'
            properties['半径'] = radius
            properties['位置'] = (loc.X(), loc.Y(), loc.Z())
        
        elif curve_type == GeomAbs_Ellipse:
            ellipse = curve_adaptor.Ellipse()
            major_radius = ellipse.MajorRadius()
            minor_radius = ellipse.MinorRadius()
            properties['类型'] = '椭圆'
            properties['长半轴'] = major_radius
            properties['短半轴'] = minor_radius
        
        elif curve_type == GeomAbs_BSplineCurve:
            # 检查是否是类似螺旋线的B样条曲线
            properties['类型'] = 'B样条曲线'
            
            # 通过采样点检查是否是螺旋型的
            try:
                # 获取参数范围
                first_param = curve_adaptor.FirstParameter()
                last_param = curve_adaptor.LastParameter()
                
                # 采样多个点
                sample_points = []
                for i in range(10):
                    param = first_param + (last_param - first_param) * i / 9
                    point = curve_adaptor.Value(param)
                    sample_points.append((point.X(), point.Y(), point.Z()))
                
                # 检查是否有类似螺旋线的特征（Z值递增且XY坐标旋转）
                if len(sample_points) >= 3:
                    is_spiral = True
                    z_increasing = sample_points[-1][2] > sample_points[0][2]
                    
                    # 检查XY平面投影是否类似圆
                    center_x = sum(p[0] for p in sample_points) / len(sample_points)
                    center_y = sum(p[1] for p in sample_points) / len(sample_points)
                    
                    # 计算到中心的距离，应该近似相等
                    distances = [((p[0] - center_x)**2 + (p[1] - center_y)**2)**0.5 for p in sample_points]
                    avg_distance = sum(distances) / len(distances)
                    
                    # 如果距离变化不大，可能是螺旋线
                    is_circular = all(abs(d - avg_distance) / avg_distance < 0.2 for d in distances)
                    
                    if z_increasing and is_circular:
                        properties['可能是螺旋线'] = True
                        properties['半径估计'] = avg_distance
                        
                        # 估算螺距
                        z_diff = sample_points[-1][2] - sample_points[0][2]
                        # 假设采样了大约一圈
                        properties['螺距估计'] = z_diff
            except:
                pass
        
        return properties
    
    @staticmethod
    def calculate_surface_area(face):
        """
        计算面的表面积
        
        Args:
            face: OCC面对象
            
        Returns:
            float: 面的表面积
        """
        props = GProp_GProps()
        # Use the new static method instead of the deprecated function
        brepgprop.SurfaceProperties(face, props)
        return props.Mass()
    
    @staticmethod
    def calculate_volume(solid):
        """
        计算实体的体积
        
        Args:
            solid: OCC实体对象
            
        Returns:
            float: 实体的体积
        """
        props = GProp_GProps()
        # Use the new static method instead of the deprecated function
        brepgprop.VolumeProperties(solid, props)
        return props.Mass()
    
    @staticmethod
    def check_point_in_shape(point, shape):
        """
        检查点是否在形状内部
        
        Args:
            point: gp_Pnt点对象
            shape: 要检查的形状
            
        Returns:
            bool: 点是否在形状内部
        """
        try:
            # 创建点的顶点
            vertex = BRepBuilderAPI_MakeVertex(point).Shape()
            
            # 使用分类器检查点是否在实体内部
            classifier = BRepClass3d_SolidClassifier(shape, point, 1e-3)
            state = classifier.State()
            
            # 如果状态为IN，则点在内部
            return state == TopAbs_IN
        except:
            return False