#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
圆角和倒角特征分析模块：专门用于识别和分析圆角与倒角特征
"""

import logging
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface
from OCC.Core.GeomAbs import GeomAbs_Torus, GeomAbs_Plane, GeomAbs_Cone
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.TopAbs import TopAbs_EDGE
from OCC.Core.TopoDS import topods
from OCC.Extend.TopologyUtils import TopologyExplorer

from id2.feature_analyzer_geometry import GeometryAnalyzer

logger = logging.getLogger(__name__)

class FilletChamferAnalyzer:
    """圆角和倒角特征分析类"""
    
    @staticmethod
    def is_fillet(face):
        """
        判断一个面是否是圆角
        
        Args:
            face: 候选面
            
        Returns:
            bool: 是否为圆角
        """
        # 圆角通常是环面(torus)
        surf = BRepAdaptor_Surface(face)
        if surf.GetType() != GeomAbs_Torus:
            return False
            
        # 获取面积
        area = GeometryAnalyzer.calculate_surface_area(face)
        
        # 获取环面参数
        torus = surf.Torus()
        major_radius = torus.MajorRadius()
        minor_radius = torus.MinorRadius()
        
        # 环面的理论面积
        # 实际圆角通常只是环面的一部分，而不是完整环面
        theoretical_area = 4 * 3.14159 * 3.14159 * major_radius * minor_radius
        
        # 如果面积远小于理论面积，且次半径较小，可能是圆角
        if area < 0.5 * theoretical_area and minor_radius < major_radius * 0.5:
            return True
            
        return True  # 简化判断，所有环面暂时都认为是圆角
    
    @staticmethod
    def is_chamfer(face):
        """
        判断一个面是否是倒角
        
        Args:
            face: 候选面
            
        Returns:
            bool: 是否为倒角
        """
        # 倒角通常是一个平面或锥面，且面积较小
        surf = BRepAdaptor_Surface(face)
        if surf.GetType() not in [GeomAbs_Plane, GeomAbs_Cone]:
            return False
        
        # 获取面积
        area = GeometryAnalyzer.calculate_surface_area(face)
        
        # 检查面的边缘形状，倒角通常有一些特定边缘模式
        edge_count = 0
        explorer = TopExp_Explorer(face, TopAbs_EDGE)
        while explorer.More():
            edge_count += 1
            explorer.Next()
        
        # 简化判断：如果是平面或锥面，且面积较小，边缘较少，可能是倒角
        if area < 10.0 and edge_count <= 4:
            return True
        
        return False
    
    @staticmethod
    def analyze_fillet(face):
        """
        分析圆角特征
        
        Args:
            face: 圆角面
            
        Returns:
            dict: 圆角属性
        """
        properties = {}
        
        surf = BRepAdaptor_Surface(face)
        if surf.GetType() == GeomAbs_Torus:
            torus = surf.Torus()
            major_radius = torus.MajorRadius()
            minor_radius = torus.MinorRadius()
            loc = torus.Location()
            axis = torus.Axis()
            
            properties['主半径'] = major_radius
            properties['次半径'] = minor_radius  # 圆角半径
            properties['半径'] = minor_radius  # 便于访问
            properties['位置'] = (loc.X(), loc.Y(), loc.Z())
            properties['方向'] = (axis.Direction().X(), axis.Direction().Y(), axis.Direction().Z())
            
            # 计算面积
            properties['面积'] = GeometryAnalyzer.calculate_surface_area(face)
            
            # 尝试确定圆角类型（外圆角或内圆角）
            # 这需要更复杂的拓扑分析
            properties['类型'] = '外圆角'  # 默认
        
        return properties
    
    @staticmethod
    def analyze_chamfer(face):
        """
        分析倒角特征
        
        Args:
            face: 倒角面
            
        Returns:
            dict: 倒角属性
        """
        properties = {}
        
        surf = BRepAdaptor_Surface(face)
        
        # 获取基本属性
        properties['面积'] = GeometryAnalyzer.calculate_surface_area(face)
        
        if surf.GetType() == GeomAbs_Plane:
            plane = surf.Plane()
            loc = plane.Location()
            axis = plane.Axis()
            
            properties['位置'] = (loc.X(), loc.Y(), loc.Z())
            properties['方向'] = (axis.Direction().X(), axis.Direction().Y(), axis.Direction().Z())
            properties['类型'] = '平面倒角'
            
            # 尝试计算倒角尺寸
            # 这需要分析面的边缘
            explorer = TopologyExplorer(face)
            edge_lengths = []
            
            for edge in explorer.edges():
                try:
                    # 获取边的长度
                    properties_edge = GProp_GProps()
                    BRepGProp.LinearProperties(edge, properties_edge)
                    length = properties_edge.Mass()
                    edge_lengths.append(length)
                except:
                    pass
            
            if edge_lengths:
                properties['边长'] = edge_lengths
                properties['最小边长'] = min(edge_lengths)  # 可能是倒角尺寸
        
        elif surf.GetType() == GeomAbs_Cone:
            cone = surf.Cone()
            radius = cone.RefRadius()
            semi_angle = cone.SemiAngle()
            loc = cone.Location()
            axis = cone.Axis()
            
            properties['半径'] = radius
            properties['半角'] = semi_angle
            properties['位置'] = (loc.X(), loc.Y(), loc.Z())
            properties['方向'] = (axis.Direction().X(), axis.Direction().Y(), axis.Direction().Z())
            properties['类型'] = '锥面倒角'
            
            # 可以从半角计算倒角角度
            angle_degrees = semi_angle * 180 / 3.14159
            properties['角度'] = angle_degrees
            
            # 如果角度接近45度，可能是标准45度倒角
            if abs(angle_degrees - 45) < 5:
                properties['标准角度'] = 45
        
        return properties