#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
凸台特征分析模块 - 修复版
改进内容：
1. 修复双重@staticmethod装饰器
2. 增强凸起/凹陷判断避免与内六角槽混淆
3. 增加面积阈值和置信度检查
"""

import logging
import math
from typing import Dict, List, Optional, Tuple

from OCC.Core.TopAbs import TopAbs_FACE, TopAbs_EDGE, TopAbs_VERTEX
from OCC.Core.TopoDS import topods
from OCC.Core.TopExp import TopExp_Explorer
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface, BRepAdaptor_Curve
from OCC.Core.GeomAbs import GeomAbs_Plane, GeomAbs_Cylinder, GeomAbs_Cone
from OCC.Core.Bnd import Bnd_Box
from OCC.Core.BRepBndLib import brepbndlib
from OCC.Core.GProp import GProp_GProps
from OCC.Core.BRepGProp import brepgprop
from OCC.Core.gp import gp_Pnt, gp_Vec, gp_Dir
from OCC.Core.BRepClass3d import BRepClass3d_SolidClassifier
from OCC.Core.TopAbs import TopAbs_IN, TopAbs_OUT, TopAbs_ON
from OCC.Core.Precision import precision

logger = logging.getLogger(__name__)


class BossAnalyzer:
    """凸台特征分析器 - 修复版"""
    
    # 配置参数
    MIN_CONFIDENCE = 0.75  # 最小置信度阈值
    MAX_AREA_RATIO = 0.15  # 最大面积比例（特征面积/总面积）
    MIN_SIDEWALLS = 1  # 最少侧壁数量
    
    @staticmethod
    def is_boss(face, shape) -> bool:
        """
        判断一个面是否属于凸台特征 - 修复版
        
        改进:
        1. 使用多点采样判断凸起/凹陷
        2. 增加面积阈值检查
        3. 增加置信度要求
        4. 排除内六角槽等凹特征
        """
        try:
            surf = BRepAdaptor_Surface(face)
            
            # 凸台的顶面通常是平面
            if surf.GetType() != GeomAbs_Plane:
                return False
            
            # ✅ 修复1: 使用多点采样判断是否凸起
            is_protruding, confidence = BossAnalyzer._check_protrusion(face, shape)
            
            if not is_protruding:
                logger.debug(f"不是凸起特征或置信度不足: {confidence:.2f}")
                return False
            
            if confidence < BossAnalyzer.MIN_CONFIDENCE:
                logger.debug(f"置信度过低: {confidence:.2f}")
                return False
            
            # ✅ 修复2: 检查面积比例
            face_area = BossAnalyzer._calculate_surface_area(face)
            total_area = BossAnalyzer._calculate_total_area(shape)
            
            if total_area > 0:
                area_ratio = face_area / total_area
                if area_ratio > BossAnalyzer.MAX_AREA_RATIO:
                    logger.debug(f"面积比例过大({area_ratio:.2%})，可能是基体")
                    return False
            
            # 获取相邻面分析整体结构
            adjacent_faces = BossAnalyzer._get_adjacent_faces(face, shape)
            
            if len(adjacent_faces) < BossAnalyzer.MIN_SIDEWALLS:
                return False
            
            # ✅ 修复3: 排除内六角槽（检查相邻面数量和角度）
            if BossAnalyzer._looks_like_hex_socket(face, adjacent_faces):
                logger.debug("检测到类似内六角槽的结构，排除")
                return False
            
            # 分析几何特征
            geometry_analysis = BossAnalyzer._analyze_boss_geometry(face, adjacent_faces, shape)
            
            if not geometry_analysis:
                return False
            
            # 尺寸合理性检查
            height = geometry_analysis.get('height', 0)
            top_area = geometry_analysis.get('top_area', 0)
            base_area = geometry_analysis.get('base_area', 0)
            
            if height > 0 and top_area > 0:
                characteristic_length = math.sqrt(top_area)
                
                # 高度约束
                if not (characteristic_length * 0.1 < height < characteristic_length * 5):
                    logger.debug(f"高度不合理: h={height:.2f}, L={characteristic_length:.2f}")
                    return False
                
                # 顶面/底面比例检查
                if base_area > 0:
                    area_ratio = top_area / base_area
                    if area_ratio > 1.5:
                        logger.debug(f"顶面/底面比例不合理: {area_ratio:.2f}")
                        return False
                
                # 检查是否有典型的凸台侧壁
                if BossAnalyzer._has_boss_sidewalls(face, adjacent_faces):
                    return True
            
            return False
            
        except Exception as e:
            logger.warning(f"凸台特征判断时出错: {str(e)}")
            return False
    
    @staticmethod
    def _check_protrusion(face, shape) -> Tuple[bool, float]:
        """
        检查面是否是凸起特征（向外突出）
        
        方法：沿法向量方向多点采样，检查偏移点是否在形状外部
        
        Returns:
            Tuple[是否凸起, 置信度]
        """
        try:
            # 获取面的中心和法向量
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            face_center = props.CentreOfMass()
            face_area = props.Mass()
            
            surf = BRepAdaptor_Surface(face)
            plane = surf.Plane()
            normal = plane.Axis().Direction()
            normal_vec = gp_Vec(normal.X(), normal.Y(), normal.Z())
            
            # 归一化
            mag = normal_vec.Magnitude()
            if mag < 1e-10:
                return False, 0.0
            normal_vec.Divide(mag)
            
            # 创建正交向量用于平面内采样
            perp1 = BossAnalyzer._create_perpendicular_vector(normal_vec)
            perp2 = normal_vec.Crossed(perp1)
            if perp2.Magnitude() > 1e-10:
                perp2.Divide(perp2.Magnitude())
            
            # 多点采样
            outward_count = 0
            inward_count = 0
            total_tests = 0
            
            # 采样参数
            offsets = [0.5, 1.0, 2.0, 3.0]  # 偏移距离
            radial_samples = 8
            radial_offset = math.sqrt(face_area / math.pi) * 0.3
            
            for offset in offsets:
                for i in range(radial_samples):
                    angle = 2 * math.pi * i / radial_samples
                    
                    # 计算采样点基准位置
                    rx = radial_offset * math.cos(angle)
                    ry = radial_offset * math.sin(angle)
                    
                    base_x = face_center.X() + perp1.X() * rx + perp2.X() * ry
                    base_y = face_center.Y() + perp1.Y() * rx + perp2.Y() * ry
                    base_z = face_center.Z() + perp1.Z() * rx + perp2.Z() * ry
                    
                    # 沿法向量正向偏移的测试点
                    test_point = gp_Pnt(
                        base_x + normal_vec.X() * offset,
                        base_y + normal_vec.Y() * offset,
                        base_z + normal_vec.Z() * offset
                    )
                    
                    classifier = BRepClass3d_SolidClassifier()
                    classifier.Load(shape)
                    classifier.Perform(test_point, precision.Confusion())
                    state = classifier.State()
                    
                    total_tests += 1
                    
                    if state == TopAbs_OUT:
                        outward_count += 1
                    elif state == TopAbs_IN:
                        inward_count += 1
            
            if total_tests == 0:
                return False, 0.0
            
            outward_ratio = outward_count / total_tests
            inward_ratio = inward_count / total_tests
            
            # 凸起特征：法向量方向（外侧）大部分点在形状外
            if outward_ratio > 0.7:
                return True, outward_ratio
            
            # 凹陷特征：法向量方向大部分点在形状内
            if inward_ratio > 0.7:
                return False, inward_ratio
            
            return False, max(outward_ratio, inward_ratio)
            
        except Exception as e:
            logger.warning(f"检查凸起时出错: {str(e)}")
            return False, 0.0
    
    @staticmethod
    def _looks_like_hex_socket(face, adjacent_faces) -> bool:
        """
        检查是否看起来像内六角槽
        
        内六角槽特征：
        - 6个相邻平面
        - 相邻面法向量呈60°角分布
        """
        if len(adjacent_faces) < 5:
            return False
        
        try:
            # 统计平面相邻面
            plane_count = 0
            normals = []
            
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                if adj_surf.GetType() == GeomAbs_Plane:
                    plane_count += 1
                    plane = adj_surf.Plane()
                    normal = plane.Axis().Direction()
                    normals.append((normal.X(), normal.Y(), normal.Z()))
            
            # 如果有5-6个平面相邻面，可能是内六角
            if plane_count >= 5:
                # 检查法向量是否呈60°分布
                import numpy as np
                angle_60_count = 0
                
                for i in range(len(normals)):
                    for j in range(i + 1, len(normals)):
                        n1 = np.array(normals[i])
                        n2 = np.array(normals[j])
                        dot = abs(np.dot(n1, n2))
                        dot = np.clip(dot, 0, 1)
                        angle = np.degrees(np.arccos(dot))
                        
                        if 55 <= angle <= 65:  # 约60°
                            angle_60_count += 1
                
                # 内六角有多对60°夹角
                if angle_60_count >= 3:
                    return True
            
            return False
            
        except Exception as e:
            logger.warning(f"检查内六角槽时出错: {str(e)}")
            return False
    
    @staticmethod
    def analyze_boss(face, shape) -> Dict:
        """
        分析凸台特征的属性
        """
        properties = {
            '类型': '凸台',
            '高度': 0.0,
            '顶面形状': '未知',
            '顶面尺寸': {},
            '底面尺寸': {},
            '锥度': 0.0,
            '顶面积': 0.0,
            '底面积': 0.0,
            '体积': 0.0,
            '质心': None,
            '用途': '未知'
        }
        
        try:
            adjacent_faces = BossAnalyzer._get_adjacent_faces(face, shape)
            geometry = BossAnalyzer._analyze_boss_geometry(face, adjacent_faces, shape)
            
            if geometry:
                properties['高度'] = round(geometry['height'], 3)
                properties['顶面积'] = round(geometry['top_area'], 3)
                properties['底面积'] = round(geometry['base_area'], 3)
                properties['锥度'] = round(geometry.get('taper', 0), 3)
                
                top_shape = BossAnalyzer._analyze_face_shape(face)
                properties['顶面形状'] = top_shape['shape']
                properties['顶面尺寸'] = top_shape['dimensions']
                
                base_dimensions = BossAnalyzer._estimate_base_dimensions(face, adjacent_faces, geometry)
                properties['底面尺寸'] = base_dimensions
            
            volume = BossAnalyzer._estimate_boss_volume(face, adjacent_faces, geometry)
            properties['体积'] = round(volume, 3)
            
            centroid = BossAnalyzer._get_face_center(face)
            if centroid:
                properties['质心'] = (round(centroid.X(), 3), round(centroid.Y(), 3), round(centroid.Z(), 3))
            
            purpose = BossAnalyzer._analyze_boss_purpose(face, adjacent_faces, geometry)
            properties['用途'] = purpose
            
            return properties
            
        except Exception as e:
            logger.error(f"分析凸台特征时出错: {str(e)}")
            return {'类型': '凸台', '错误': str(e)}
    
    @staticmethod
    def _get_adjacent_faces(face, shape) -> List:
        """获取与给定面相邻的面"""
        adjacent_faces = []
        
        try:
            edge_explorer = TopExp_Explorer(face, TopAbs_EDGE)
            face_edges = []
            
            while edge_explorer.More():
                edge = topods.Edge(edge_explorer.Current())
                face_edges.append(edge)
                edge_explorer.Next()
            
            for edge in face_edges:
                face_explorer = TopExp_Explorer(shape, TopAbs_FACE)
                
                while face_explorer.More():
                    other_face = topods.Face(face_explorer.Current())
                    
                    if not face.IsSame(other_face):
                        other_face_edge_explorer = TopExp_Explorer(other_face, TopAbs_EDGE)
                        
                        while other_face_edge_explorer.More():
                            other_edge = topods.Edge(other_face_edge_explorer.Current())
                            
                            if edge.IsSame(other_edge):
                                if other_face not in adjacent_faces:
                                    adjacent_faces.append(other_face)
                                break
                            
                            other_face_edge_explorer.Next()
                    
                    face_explorer.Next()
            
            return adjacent_faces
            
        except Exception as e:
            logger.warning(f"获取相邻面时出错: {str(e)}")
            return []
    
    @staticmethod
    def _analyze_boss_geometry(face, adjacent_faces, shape) -> Optional[Dict]:
        """分析凸台的几何特征"""
        try:
            top_area = BossAnalyzer._calculate_surface_area(face)
            height, base_area = BossAnalyzer._estimate_height_and_base_area(face, adjacent_faces, shape)
            taper = BossAnalyzer._calculate_taper(face, adjacent_faces, height)
            
            return {
                'height': height,
                'top_area': top_area,
                'base_area': base_area,
                'taper': taper
            }
            
        except Exception as e:
            logger.warning(f"分析凸台几何时出错: {str(e)}")
            return None
    
    @staticmethod
    def _estimate_height_and_base_area(face, adjacent_faces, shape) -> Tuple[float, float]:
        """估算凸台高度和底面积"""
        try:
            face_center = BossAnalyzer._get_face_center(face)
            if not face_center:
                return 0.0, 0.0
            
            surf = BRepAdaptor_Surface(face)
            u_mid = (surf.FirstUParameter() + surf.LastUParameter()) / 2
            v_mid = (surf.FirstVParameter() + surf.LastVParameter()) / 2
            
            pnt = gp_Pnt()
            vec_u = gp_Vec()
            vec_v = gp_Vec()
            surf.D1(u_mid, v_mid, pnt, vec_u, vec_v)
            
            normal = vec_u.Crossed(vec_v)
            if normal.Magnitude() > 0:
                normal.Normalize()
            else:
                return 0.0, 0.0
            
            max_distance = 0.0
            base_area = 0.0
            
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                
                if adj_surf.GetType() == GeomAbs_Plane:
                    adj_plane = adj_surf.Plane()
                    adj_normal = adj_plane.Axis().Direction()
                    
                    dot_product = abs(normal.X() * adj_normal.X() + 
                                     normal.Y() * adj_normal.Y() + 
                                     normal.Z() * adj_normal.Z())
                    
                    if dot_product > 0.9:  # 平行面
                        distance = abs(surf.Plane().Distance(adj_plane.Location()))
                        if distance > max_distance:
                            max_distance = distance
                            base_area = BossAnalyzer._calculate_surface_area(adj_face)
            
            return max_distance, base_area
            
        except Exception as e:
            logger.warning(f"估算高度和底面积时出错: {str(e)}")
            return 0.0, 0.0
    
    @staticmethod
    def _calculate_taper(face, adjacent_faces, height) -> float:
        """计算锥度"""
        try:
            if not adjacent_faces or height <= 0:
                return 0.0
            
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                
                if adj_surf.GetType() == GeomAbs_Cone:
                    cone = adj_surf.Cone()
                    semi_angle = cone.SemiAngle()
                    return math.degrees(semi_angle)
            
            return 0.0
            
        except Exception as e:
            logger.warning(f"计算锥度时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _has_boss_sidewalls(face, adjacent_faces) -> bool:
        """检查是否有典型的凸台侧壁"""
        try:
            if not adjacent_faces:
                return False
            
            surf = BRepAdaptor_Surface(face)
            face_normal = surf.Plane().Axis().Direction()
            
            sidewall_count = 0
            
            for adj_face in adjacent_faces:
                adj_surf = BRepAdaptor_Surface(adj_face)
                
                if adj_surf.GetType() in [GeomAbs_Plane, GeomAbs_Cylinder, GeomAbs_Cone]:
                    sidewall_count += 1
            
            return sidewall_count >= 1
            
        except Exception as e:
            logger.warning(f"检查侧壁时出错: {str(e)}")
            return False
    
    @staticmethod
    def _analyze_face_shape(face) -> Dict:
        """分析面的形状"""
        try:
            result = {'shape': '未知', 'dimensions': {}}
            
            bbox = BossAnalyzer._get_face_bounding_box(face)
            if not bbox:
                return result
            
            dx, dy, dz = bbox['dx'], bbox['dy'], bbox['dz']
            dimensions = sorted([dx, dy, dz], reverse=True)
            dim1, dim2 = dimensions[0], dimensions[1]
            
            ratio = dim1 / dim2 if dim2 > 0 else float('inf')
            
            if ratio < 1.2:
                result['shape'] = '圆形或正方形'
                result['dimensions'] = {'直径/边长': round(dim1, 3)}
            elif ratio < 2.0:
                result['shape'] = '矩形'
                result['dimensions'] = {'长度': round(dim1, 3), '宽度': round(dim2, 3)}
            else:
                result['shape'] = '长条形'
                result['dimensions'] = {'长度': round(dim1, 3), '宽度': round(dim2, 3)}
            
            return result
            
        except Exception as e:
            logger.warning(f"分析面形状时出错: {str(e)}")
            return {'shape': '未知', 'dimensions': {}}
    
    @staticmethod
    def _estimate_base_dimensions(face, adjacent_faces, geometry) -> Dict:
        """估算底面尺寸"""
        try:
            top_bbox = BossAnalyzer._get_face_bounding_box(face)
            if not top_bbox:
                return {}
            
            taper = geometry.get('taper', 0) if geometry else 0
            height = geometry.get('height', 0) if geometry else 0
            
            if taper > 0 and height > 0:
                taper_rad = math.radians(taper)
                expansion = height * math.tan(taper_rad)
                
                return {
                    '长度': round(top_bbox['dx'] + 2 * expansion, 3),
                    '宽度': round(top_bbox['dy'] + 2 * expansion, 3)
                }
            else:
                return {
                    '长度': round(top_bbox['dx'], 3),
                    '宽度': round(top_bbox['dy'], 3)
                }
            
        except Exception as e:
            logger.warning(f"估算底面尺寸时出错: {str(e)}")
            return {}
    
    @staticmethod
    def _estimate_boss_volume(face, adjacent_faces, geometry) -> float:
        """估算凸台体积"""
        try:
            if not geometry:
                return 0.0
            
            height = geometry.get('height', 0)
            top_area = geometry.get('top_area', 0)
            base_area = geometry.get('base_area', 0)
            
            if height > 0 and top_area > 0 and base_area > 0:
                volume = height / 3 * (top_area + base_area + math.sqrt(top_area * base_area))
                return volume
            elif height > 0 and top_area > 0:
                return top_area * height
            
            return 0.0
            
        except Exception as e:
            logger.warning(f"估算凸台体积时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _analyze_boss_purpose(face, adjacent_faces, geometry) -> str:
        """分析凸台的可能用途"""
        try:
            if not geometry:
                return "未知"
            
            top_area = geometry.get('top_area', 0)
            height = geometry.get('height', 0)
            
            diameter = math.sqrt(4 * top_area / math.pi) if top_area > 0 else 0
            
            if diameter > 0 and height > 0:
                aspect_ratio = height / diameter
                
                if aspect_ratio > 2:
                    return "定位柱"
                elif aspect_ratio > 0.5:
                    return "连接凸台"
                elif aspect_ratio < 0.2:
                    return "加强凸台"
                else:
                    return "通用凸台"
            
            return "通用凸台"
            
        except Exception as e:
            logger.warning(f"分析凸台用途时出错: {str(e)}")
            return "未知"
    
    @staticmethod
    def _create_perpendicular_vector(vec: gp_Vec) -> gp_Vec:
        """创建垂直向量"""
        if abs(vec.X()) < 0.9:
            ref = gp_Vec(1, 0, 0)
        else:
            ref = gp_Vec(0, 1, 0)
        
        perp = vec.Crossed(ref)
        mag = perp.Magnitude()
        if mag > 1e-10:
            perp.Divide(mag)
        
        return perp
    
    @staticmethod
    def _get_face_bounding_box(face) -> Optional[Dict]:
        """获取面的边界框"""
        try:
            bbox = Bnd_Box()
            brepbndlib.Add(face, bbox)
            xmin, ymin, zmin, xmax, ymax, zmax = bbox.Get()
            
            return {
                'xmin': xmin, 'ymin': ymin, 'zmin': zmin,
                'xmax': xmax, 'ymax': ymax, 'zmax': zmax,
                'dx': xmax - xmin,
                'dy': ymax - ymin,
                'dz': zmax - zmin
            }
            
        except Exception as e:
            logger.warning(f"获取边界框时出错: {str(e)}")
            return None
    
    @staticmethod
    def _calculate_surface_area(face) -> float:
        """计算面的表面积"""
        try:
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            return props.Mass()
        except Exception as e:
            logger.warning(f"计算面积时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _calculate_total_area(shape) -> float:
        """计算整体形状的总表面积"""
        try:
            total_area = 0.0
            explorer = TopExp_Explorer(shape, TopAbs_FACE)
            
            while explorer.More():
                face = topods.Face(explorer.Current())
                props = GProp_GProps()
                brepgprop.SurfaceProperties(face, props)
                total_area += props.Mass()
                explorer.Next()
            
            return total_area
        except Exception as e:
            logger.warning(f"计算总面积时出错: {str(e)}")
            return 0.0
    
    @staticmethod
    def _get_face_center(face) -> Optional[gp_Pnt]:
        """获取面的中心点"""
        try:
            props = GProp_GProps()
            brepgprop.SurfaceProperties(face, props)
            return props.CentreOfMass()
        except Exception as e:
            logger.warning(f"获取面中心时出错: {str(e)}")
            return None
