#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
特征分析主模块：整合所有特征分析子模块
更新版本：增加了筋、沟槽、凸台、凹槽特征分析
"""

import logging
from OCC.Core.BRepAdaptor import BRepAdaptor_Surface
from OCC.Core.GeomAbs import GeomAbs_Cylinder, GeomAbs_Torus, GeomAbs_Plane, GeomAbs_Cone

from id2.feature_base import 特征类型
from id2.feature_analyzer_geometry import GeometryAnalyzer
from id2.feature_analyzer_hole import HoleAnalyzer
from id2.feature_analyzer_thread import ThreadAnalyzer
from id2.feature_analyzer_fillet import FilletChamferAnalyzer
from id2.feature_analyzer_rib import RibAnalyzer
from id2.feature_analyzer_groove import GrooveAnalyzer
from id2.feature_analyzer_boss import BossAnalyzer
from id2.feature_analyzer_pocket import PocketAnalyzer
from id2.feature_analyzer_hexsocket import HexSocketAnalyzer
from id2.feature_semantic_analyzer import SemanticFeatureAnalyzer

logger = logging.getLogger(__name__)

class FeatureAnalyzer:
    """特征分析类，整合所有特征分析功能"""
    
    # ========== 基础几何分析 ==========
    @staticmethod
    def analyze_surface(face):
        """
        分析面的表面类型
        
        Args:
            face: OCC面对象
            
        Returns:
            tuple: (表面类型, 属性字典)
        """
        return GeometryAnalyzer.analyze_surface(face)
    
    @staticmethod
    def analyze_edge(edge):
        """
        分析边的类型
        
        Args:
            edge: OCC边对象
            
        Returns:
            dict: 边属性字典
        """
        return GeometryAnalyzer.analyze_edge(edge)
    
    @staticmethod
    def calculate_surface_area(face):
        """
        计算面的表面积
        
        Args:
            face: OCC面对象
            
        Returns:
            float: 面的表面积
        """
        return GeometryAnalyzer.calculate_surface_area(face)
    
    @staticmethod
    def calculate_volume(solid):
        """
        计算实体的体积
        
        Args:
            solid: OCC实体对象
            
        Returns:
            float: 实体的体积
        """
        return GeometryAnalyzer.calculate_volume(solid)
    
    # ========== 孔特征分析 ==========
    @staticmethod
    def is_hole(cylindrical_face, shape):
        """
        判断一个圆柱面是否构成孔
        
        Args:
            cylindrical_face: 圆柱面
            shape: 整体形状
            
        Returns:
            bool: 是否为孔
        """
        return HoleAnalyzer.is_hole(cylindrical_face, shape)
    
    @staticmethod
    def analyze_hole(cylindrical_face, shape):
        """
        分析孔的属性
        
        Args:
            cylindrical_face: 孔的圆柱面
            shape: 整体形状
            
        Returns:
            dict: 孔的属性
        """
        return HoleAnalyzer.analyze_hole(cylindrical_face, shape)
    
    @staticmethod
    def calculate_hole_depth(cylindrical_face, shape):
        """
        计算孔的深度
        
        Args:
            cylindrical_face: 孔的圆柱面
            shape: 整体形状
            
        Returns:
            float: 孔的深度
        """
        return HoleAnalyzer.calculate_hole_depth(cylindrical_face, shape)
    
    # ========== 螺纹特征分析 ==========
    @staticmethod
    def is_thread(cylindrical_face, shape):
        """
        判断一个圆柱面是否构成螺纹
        
        Args:
            cylindrical_face: 圆柱面
            shape: 整体形状
            
        Returns:
            bool: 是否为螺纹
        """
        return ThreadAnalyzer.is_thread(cylindrical_face, shape)
    
    @staticmethod
    def analyze_thread(cylindrical_face, shape):
        """
        分析螺纹特征
        
        Args:
            cylindrical_face: 螺纹的圆柱面
            shape: 整体形状
            
        Returns:
            dict: 螺纹属性
        """
        return ThreadAnalyzer.analyze_thread(cylindrical_face, shape)
    
    # ========== 圆角和倒角特征分析 ==========
    @staticmethod
    def is_fillet(face):
        """
        判断一个面是否是圆角
        
        Args:
            face: 候选面
            
        Returns:
            bool: 是否为圆角
        """
        return FilletChamferAnalyzer.is_fillet(face)
    
    @staticmethod
    def is_chamfer(face):
        """
        判断一个面是否是倒角
        
        Args:
            face: 候选面
            
        Returns:
            bool: 是否为倒角
        """
        return FilletChamferAnalyzer.is_chamfer(face)
    
    @staticmethod
    def analyze_fillet(face):
        """
        分析圆角特征
        
        Args:
            face: 圆角面
            
        Returns:
            dict: 圆角属性
        """
        return FilletChamferAnalyzer.analyze_fillet(face)
    
    @staticmethod
    def analyze_chamfer(face):
        """
        分析倒角特征
        
        Args:
            face: 倒角面
            
        Returns:
            dict: 倒角属性
        """
        return FilletChamferAnalyzer.analyze_chamfer(face)
    
    # ========== 筋特征分析 ==========
    @staticmethod
    def is_rib(face, shape):
        """
        判断一个面是否属于筋特征
        
        Args:
            face: 候选面
            shape: 整体形状
            
        Returns:
            bool: 是否为筋
        """
        return RibAnalyzer.is_rib(face, shape)
    
    @staticmethod
    def analyze_rib(face, shape):
        """
        分析筋特征的属性
        
        Args:
            face: 筋的面
            shape: 整体形状
            
        Returns:
            dict: 筋的属性
        """
        return RibAnalyzer.analyze_rib(face, shape)
    
    # ========== 沟槽特征分析 ==========
    @staticmethod
    def is_groove(face, shape):
        """
        判断一个面是否属于沟槽特征
        
        Args:
            face: 候选面
            shape: 整体形状
            
        Returns:
            bool: 是否为沟槽
        """
        return GrooveAnalyzer.is_groove(face, shape)
    
    @staticmethod
    def analyze_groove(face, shape):
        """
        分析沟槽特征的属性
        
        Args:
            face: 沟槽的底面
            shape: 整体形状
            
        Returns:
            dict: 沟槽的属性
        """
        return GrooveAnalyzer.analyze_groove(face, shape)
    
    # ========== 凸台特征分析 ==========
    @staticmethod
    def is_boss(face, shape):
        """
        判断一个面是否属于凸台特征
        
        Args:
            face: 候选面
            shape: 整体形状
            
        Returns:
            bool: 是否为凸台
        """
        return BossAnalyzer.is_boss(face, shape)
    
    @staticmethod
    def analyze_boss(face, shape):
        """
        分析凸台特征的属性
        
        Args:
            face: 凸台的顶面
            shape: 整体形状
            
        Returns:
            dict: 凸台的属性
        """
        return BossAnalyzer.analyze_boss(face, shape)
    
    # ========== 凹槽特征分析 ==========
    @staticmethod
    def is_pocket(face, shape):
        """
        判断一个面是否属于凹槽特征
        
        Args:
            face: 候选面（通常是底面）
            shape: 整体形状
            
        Returns:
            bool: 是否为凹槽
        """
        return PocketAnalyzer.is_pocket(face, shape)
    
    @staticmethod
    def analyze_pocket(face, shape):
        """
        分析凹槽特征的属性
        
        Args:
            face: 凹槽的底面
            shape: 整体形状
            
        Returns:
            dict: 凹槽的属性
        """
        return PocketAnalyzer.analyze_pocket(face, shape)
    
    # ========== 槽特征分析（占位，需要实现） ==========
    @staticmethod
    def is_slot(face, shape):
        """
        判断一个面是否属于槽特征
        
        Args:
            face: 候选面
            shape: 整体形状
            
        Returns:
            bool: 是否为槽
        """
        # TODO: 实现槽特征识别
        return False
    
    @staticmethod
    def analyze_slot(face, shape):
        """
        分析槽特征的属性
        
        Args:
            face: 槽的面
            shape: 整体形状
            
        Returns:
            dict: 槽的属性
        """
        # TODO: 实现槽特征分析
        return {'类型': '槽', '状态': '未实现'}
    
    # ========== 综合特征判断 ==========
    @staticmethod
    def identify_complex_feature(face, shape):
        """
        识别复杂特征（筋、沟槽、凸台、凹槽等）
        
        Args:
            face: 候选面
            shape: 整体形状
            
        Returns:
            tuple: (特征类型, 是否识别成功, 属性字典)
        """
        try:
            # 按优先级顺序检查复杂特征
            
            # 1. 检查筋特征
            if FeatureAnalyzer.is_rib(face, shape):
                properties = FeatureAnalyzer.analyze_rib(face, shape)
                return 特征类型['筋'], True, properties
            
            # 2. 检查凸台特征
            if FeatureAnalyzer.is_boss(face, shape):
                properties = FeatureAnalyzer.analyze_boss(face, shape)
                return 特征类型['凸台'], True, properties
            
            # 3. 检查凹槽特征
            if FeatureAnalyzer.is_pocket(face, shape):
                properties = FeatureAnalyzer.analyze_pocket(face, shape)
                return 特征类型['凹槽'], True, properties
            
            # 4. 检查沟槽特征
            if FeatureAnalyzer.is_groove(face, shape):
                properties = FeatureAnalyzer.analyze_groove(face, shape)
                return 特征类型['沟槽'], True, properties
            
            # 5. 检查槽特征（待实现）
            if FeatureAnalyzer.is_slot(face, shape):
                properties = FeatureAnalyzer.analyze_slot(face, shape)
                return 特征类型['槽'], True, properties
            
            return None, False, {}
            
        except Exception as e:
            logger.warning(f"识别复杂特征时出错: {str(e)}")
            return None, False, {'错误': str(e)}
    
    # ========== 内六角特征分析 ==========
    @staticmethod
    def find_hex_sockets(shape):
        """
        查找形状中的内六角槽特征
        
        Args:
            shape: 整体形状
            
        Returns:
            List[Dict]: 内六角特征列表
        """
        return HexSocketAnalyzer.find_hex_socket_candidates(shape)
    
    @staticmethod
    def analyze_hex_socket(hex_faces, shape):
        """
        分析内六角槽特征
        
        Args:
            hex_faces: 构成内六角的面列表
            shape: 整体形状
            
        Returns:
            dict: 内六角属性
        """
        return HexSocketAnalyzer.analyze_hex_socket(hex_faces, shape)
    
    @staticmethod
    def is_hex_socket_face(face, shape):
        """
        判断一个面是否可能是内六角槽的一部分
        
        Args:
            face: 候选面
            shape: 整体形状
            
        Returns:
            bool: 是否可能是内六角面
        """
        return HexSocketAnalyzer.is_hex_socket_face(face, shape)
    
    # ========== 语义特征分析 ==========
    @staticmethod
    def analyze_semantic_features(features, shape):
        """
        分析语义特征，将基础几何组合为高级特征
        
        Args:
            features: 基础特征列表
            shape: 整体形状
            
        Returns:
            List[Dict]: 语义特征列表
        """
        return SemanticFeatureAnalyzer.analyze_semantic_features(features, shape)
    
    @staticmethod
    def analyze_cone_semantics(cone_face, shape):
        """
        分析锥面的语义含义
        
        Args:
            cone_face: 锥面
            shape: 整体形状
            
        Returns:
            dict: 语义分析结果
        """
        return SemanticFeatureAnalyzer.analyze_cone_semantics(cone_face, shape)