# PRT特征识别API服务

## 项目简介

这是一个基于FastAPI开发的PRT/STEP文件机械特征识别服务，提供RESTful API接口用于自动化提取3D模型的机械特征。

### 主要功能

- ✅ **文件上传与异步处理**：支持PRT/STEP文件上传，后台异步处理
- ✅ **特征识别**：自动识别孔、槽、圆角等机械特征
- ✅ **拓扑关系分析**：分析特征间的拓扑关系
- ✅ **任务管理**：完整的任务状态跟踪和结果管理
- ✅ **主板UID验证**：内置授权保护机制
- ✅ **RESTful API**：标准化的HTTP接口
- ✅ **自动文档**：Swagger UI和ReDoc自动生成

## 技术栈

- **Web框架**: FastAPI 0.104+
- **3D处理**: PythonOCC (OpenCASCADE)
- **异步处理**: Python AsyncIO + BackgroundTasks
- **API文档**: OpenAPI 3.0 (Swagger UI)
- **部署**: Uvicorn / Gunicorn / Docker

## 快速开始

### 1. 安装依赖

#### 方法1: 使用Conda（推荐）

```bash
# 创建conda环境
conda create -n prt_api python=3.9 -y
conda activate prt_api

# 安装pythonocc
conda install -c conda-forge pythonocc-core -y

# 安装其他依赖
pip install -r requirements.txt
```

#### 方法2: 使用Docker

```bash
# 构建镜像
docker-compose build

# 启动服务
docker-compose up -d
```

### 2. 启动服务

#### 开发模式

```bash
python prt_feature_api.py --host 0.0.0.0 --port 8000 --reload
```

#### 生产模式（多进程）

```bash
# Linux/Mac
gunicorn prt_feature_api:app -w 4 -k uvicorn.workers.UvicornWorker --bind 0.0.0.0:8000

# Windows
python prt_feature_api.py --host 0.0.0.0 --port 8000 --workers 4
```

### 3. 访问服务

- **API文档（Swagger UI）**: http://localhost:8000/docs
- **API文档（ReDoc）**: http://localhost:8000/redoc
- **健康检查**: http://localhost:8000/health

## API使用指南

### 1. 健康检查

```bash
curl http://localhost:8000/health
```

响应：
```json
{
  "status": "healthy",
  "version": "2.0.0",
  "motherboard_uid_verified": true,
  "timestamp": "2025-11-18T10:30:00"
}
```

### 2. 上传文件并提取特征

```bash
curl -X POST "http://localhost:8000/api/v1/extract/upload" \
  -F "file=@your_model.stp" \
  -F "enable_topology_relations=true" \
  -F "enable_process_planning=false"
```

响应：
```json
{
  "job_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
  "status": "pending",
  "message": "任务已创建",
  "created_at": "2025-11-18T10:30:00",
  "completed_at": null,
  "result_file": null,
  "error": null
}
```

### 3. 查询任务状态

```bash
curl http://localhost:8000/api/v1/jobs/{job_id}
```

响应：
```json
{
  "job_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
  "status": "completed",
  "message": "特征提取完成",
  "created_at": "2025-11-18T10:30:00",
  "completed_at": "2025-11-18T10:31:30",
  "result_file": "a1b2c3d4-e5f6-7890-abcd-ef1234567890/model_features.json",
  "error": null
}
```

任务状态说明：
- `pending`: 等待处理
- `processing`: 正在处理
- `completed`: 已完成
- `failed`: 失败

### 4. 下载结果

```bash
curl http://localhost:8000/api/v1/jobs/{job_id}/result -o result.json
```

### 5. 列出所有任务

```bash
curl http://localhost:8000/api/v1/jobs
```

### 6. 删除任务

```bash
curl -X DELETE http://localhost:8000/api/v1/jobs/{job_id}
```

### 7. 清理旧任务

```bash
curl -X POST "http://localhost:8000/api/v1/cleanup?max_age_hours=24"
```

## Python客户端使用

### 基本使用

```python
from prt_api_client import PRTFeatureAPIClient

# 创建客户端
client = PRTFeatureAPIClient("http://localhost:8000")

# 健康检查
health = client.check_health()
print(health)

# 上传文件
job_info = client.upload_file("model.stp", enable_topology=True)
job_id = job_info['job_id']

# 等待完成
status = client.wait_for_completion(job_id, max_wait=300)

# 获取结果
if status['status'] == 'completed':
    result = client.get_job_result(job_id, "result.json")
    print(f"特征提取完成，结果已保存")
```

### 命令行使用

```bash
# 健康检查
python prt_api_client.py --action health

# 列出任务
python prt_api_client.py --action list

# 完整测试
python prt_api_client.py --action test --file model.stp --output ./results

# 清理任务
python prt_api_client.py --action cleanup
```

## 特征提取结果格式

### JSON结构

```json
{
  "metadata": {
    "extraction_time": "2025-11-18T10:31:30",
    "version": "2.0.0-api",
    "filename": "model.stp"
  },
  "advanced_features": {
    "holes": [
      {
        "id": "hole_001",
        "type": "through_hole",
        "diameter": 10.0,
        "depth": 50.0,
        "position": [0, 0, 0],
        "direction": [0, 0, 1]
      }
    ],
    "pockets": [...],
    "fillets": [...],
    "chamfers": [...]
  },
  "basic_features": {
    "faces": [...],
    "edges": [...],
    "vertices": [...]
  },
  "topology": {
    "feature_adjacency": [...],
    "feature_groups": [...]
  }
}
```

### 主要特征类型

1. **孔特征 (Holes)**
   - 通孔 (through_hole)
   - 盲孔 (blind_hole)
   - 沉头孔 (countersunk_hole)
   - 锪孔 (counterbored_hole)

2. **槽特征 (Slots/Pockets)**
   - 直槽 (rectangular_pocket)
   - T型槽 (t_slot)
   - 燕尾槽 (dovetail_slot)

3. **圆角/倒角 (Fillets/Chamfers)**
   - 固定半径圆角
   - 可变半径圆角
   - 直线倒角

4. **拓扑关系**
   - 相邻关系 (adjacency)
   - 包含关系 (containment)
   - 相交关系 (intersection)

## 配置说明

### 环境变量

```bash
# 服务配置
export API_HOST=0.0.0.0
export API_PORT=8000
export WORKERS=4

# 日志级别
export LOG_LEVEL=INFO

# 临时目录
export TEMP_DIR=/tmp/prt_feature_api
```

### 主板UID配置

如需修改授权主板UID，编辑`prt_feature_api.py`：

```python
MOTHERBOARD_VERIFIED = verify_motherboard_uid("YOUR_UID_HERE")
```

## 性能优化

### 1. 异步处理

所有特征提取任务都在后台异步执行，不会阻塞API响应。

### 2. 多进程部署

生产环境建议使用多个worker进程：

```bash
gunicorn prt_feature_api:app -w 4 -k uvicorn.workers.UvicornWorker
```

### 3. 资源限制

建议配置：
- CPU: 4核以上
- 内存: 8GB以上
- 磁盘: 根据文件大小预留足够空间

### 4. 任务清理

定期清理旧任务释放存储空间：

```bash
# 每天清理超过24小时的任务
curl -X POST "http://localhost:8000/api/v1/cleanup?max_age_hours=24"
```

## 故障排查

### 1. 服务无法启动

**问题**: ModuleNotFoundError: No module named 'OCC'

**解决**:
```bash
conda install -c conda-forge pythonocc-core -y
```

### 2. 主板UID验证失败

**问题**: 服务未授权：主板UID验证失败

**解决**: 
- 检查是否在授权计算机上运行
- 开发环境可以修改代码跳过验证

### 3. 文件上传失败

**问题**: 文件类型不支持

**解决**: 确保文件扩展名为 `.stp`, `.step` 或 `.prt`

### 4. 任务处理超时

**问题**: 大文件处理时间过长

**解决**:
- 增加`wait_for_completion`的超时时间
- 使用轮询方式定期检查任务状态

## 安全建议

1. **生产环境部署**
   - 使用反向代理（Nginx/Apache）
   - 启用HTTPS
   - 配置速率限制
   - 实施访问控制

2. **文件上传限制**
   - 限制文件大小
   - 验证文件类型
   - 扫描恶意文件

3. **主板验证**
   - 妥善保管授权UID
   - 不要在公共代码中暴露UID

## 开发指南

### 添加新特征识别

在`id2/feature_recognition.py`中添加新的识别方法：

```python
def recognize_new_feature(self, shape):
    """识别新特征"""
    features = []
    # 特征识别逻辑
    return features
```

在`recognize_all_features`中调用：

```python
def recognize_all_features(self):
    features = {
        "new_features": self.recognize_new_feature(self.shapes[0])
    }
    return features
```

### 扩展API端点

在`prt_feature_api.py`中添加新的路由：

```python
@app.post("/api/v1/custom/endpoint")
async def custom_endpoint():
    """自定义端点"""
    return {"message": "Custom endpoint"}
```

## 许可证

本项目包含主板UID验证保护机制，仅授权计算机可运行。

## 联系方式

如有问题或建议，请联系项目维护者。

## 更新日志

### v2.0.0 (2025-11-18)
- ✨ 新增FastAPI服务
- ✨ 支持异步任务处理
- ✨ 完整的RESTful API
- ✨ 自动生成API文档
- ✨ Docker部署支持
- ✨ Python客户端工具

### v1.0.0
- 初始版本
- 基本特征识别功能
