#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
PRT文件机械特征识别API服务
提供RESTful API接口用于PRT文件特征提取
"""

import os
import sys
import json
import shutil
import tempfile
import traceback
from pathlib import Path
from datetime import datetime
from typing import Optional, List
import logging
import uuid

from fastapi import FastAPI, File, UploadFile, HTTPException, BackgroundTasks, Query
from fastapi.responses import JSONResponse, FileResponse
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel, Field
import uvicorn

# 导入原有的特征提取模块
from OCC.Core.STEPControl import STEPControl_Reader
from OCC.Core.IFSelect import IFSelect_RetDone
from id2.feature_recognition import FeatureRecognition

# 配置日志
logging.basicConfig(
    level=logging.INFO,
    format='%(asctime)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# ==================== 数据模型 ====================

class FeatureExtractionRequest(BaseModel):
    """特征提取请求模型"""
    enable_topology_relations: bool = Field(True, description="是否启用拓扑关系分析")
    enable_process_planning: bool = Field(False, description="是否启用工艺路线规划")
    material: str = Field("steel", description="材料类型")
    
    class Config:
        schema_extra = {
            "example": {
                "enable_topology_relations": True,
                "enable_process_planning": False,
                "material": "steel"
            }
        }


class JobStatus(BaseModel):
    """任务状态模型"""
    job_id: str
    status: str  # pending, processing, completed, failed
    message: str
    created_at: str
    completed_at: Optional[str] = None
    result_file: Optional[str] = None
    error: Optional[str] = None


class HealthResponse(BaseModel):
    """健康检查响应"""
    status: str
    version: str
    motherboard_uid_verified: bool
    timestamp: str


# ==================== 主板UID验证（从原代码复制）====================

def get_motherboard_serial():
    """获取主板序列号（简化版）"""
    import subprocess
    import platform
    
    try:
        if platform.system() == "Windows":
            result = subprocess.run(
                ['wmic', 'baseboard', 'get', 'serialnumber'], 
                capture_output=True, 
                text=True, 
                shell=True,
                timeout=30
            )
            
            if result.returncode == 0 and result.stdout:
                lines = result.stdout.strip().split('\n')
                for line in lines:
                    line = line.strip()
                    if line and line != 'SerialNumber':
                        return line
        else:
            # Linux/Mac系统的处理
            logger.warning("非Windows系统，跳过主板验证")
            return "DEVELOPMENT_MODE"
    except Exception as e:
        logger.error(f"获取主板序列号失败: {str(e)}")
    
    return None


def verify_motherboard_uid(required_uid="024AFQCNL8002172"):
    """验证主板UID"""
    try:
        current_uid = get_motherboard_serial()
        
        if not current_uid:
            logger.error("无法获取主板序列号")
            return False
        
        # 开发模式跳过验证
        if current_uid == "DEVELOPMENT_MODE":
            logger.warning("开发模式：跳过主板验证")
            return True
        
        current_uid = current_uid.strip().upper()
        required_uid = required_uid.strip().upper()
        
        if current_uid == required_uid:
            logger.info(f"主板UID验证成功: {current_uid}")
            return True
        else:
            logger.error(f"主板UID不匹配! 当前: {current_uid}, 要求: {required_uid}")
            return False
            
    except Exception as e:
        logger.error(f"主板UID验证出错: {str(e)}")
        return False


# ==================== 特征提取核心类 ====================

class FeatureExtractorService:
    """特征提取服务类"""
    
    def __init__(self, temp_dir: str = None):
        self.temp_dir = Path(temp_dir) if temp_dir else Path(tempfile.gettempdir()) / "prt_feature_api"
        self.temp_dir.mkdir(parents=True, exist_ok=True)
        
        # 任务存储目录
        self.jobs_dir = self.temp_dir / "jobs"
        self.jobs_dir.mkdir(exist_ok=True)
        
        # 任务状态存储
        self.job_status = {}
    
    def read_step_file(self, filepath: Path):
        """读取STEP文件"""
        reader = STEPControl_Reader()
        status = reader.ReadFile(str(filepath))
        
        if status != IFSelect_RetDone:
            raise Exception(f"无法读取STEP文件: {filepath}")
        
        reader.TransferRoots()
        shape = reader.OneShape()
        
        if shape is None:
            raise Exception(f"无法获取形状数据: {filepath}")
        
        return shape
    
    def extract_features(self, shape, output_dir: Path, filename: str):
        """提取特征"""
        logger.info("开始提取特征...")
        
        # 包装shape为列表
        shape_list = [shape]
        
        # 使用FeatureRecognition提取特征
        feature_recognition = FeatureRecognition(shape_list)
        features = feature_recognition.recognize_all_features()
        
        # 添加元数据
        features["metadata"] = {
            "extraction_time": datetime.now().isoformat(),
            "version": "2.0.0-api",
            "filename": filename
        }
        
        # 保存结果
        output_file = output_dir / f"{Path(filename).stem}_features.json"
        with open(output_file, 'w', encoding='utf-8') as f:
            json.dump(features, f, ensure_ascii=False, indent=2, default=str)
        
        logger.info(f"特征提取完成，结果保存到: {output_file}")
        return features, output_file
    
    def process_file(self, 
                     job_id: str,
                     file_path: Path,
                     enable_topology: bool = True,
                     enable_process_planning: bool = False):
        """处理单个文件"""
        try:
            # 更新状态为处理中
            self.update_job_status(job_id, "processing", "正在提取特征...")
            
            # 创建任务输出目录
            job_dir = self.jobs_dir / job_id
            job_dir.mkdir(exist_ok=True)
            
            # 读取STEP文件
            shape = self.read_step_file(file_path)
            
            # 提取特征
            features, output_file = self.extract_features(
                shape, 
                job_dir, 
                file_path.name
            )
            
            # 更新状态为完成
            self.update_job_status(
                job_id, 
                "completed", 
                "特征提取完成",
                result_file=str(output_file.relative_to(self.jobs_dir))
            )
            
            return True, output_file
            
        except Exception as e:
            error_msg = f"处理失败: {str(e)}\n{traceback.format_exc()}"
            logger.error(error_msg)
            self.update_job_status(job_id, "failed", "处理失败", error=error_msg)
            return False, error_msg
    
    def create_job(self) -> str:
        """创建新任务"""
        job_id = str(uuid.uuid4())
        self.job_status[job_id] = {
            "job_id": job_id,
            "status": "pending",
            "message": "任务已创建",
            "created_at": datetime.now().isoformat(),
            "completed_at": None,
            "result_file": None,
            "error": None
        }
        return job_id
    
    def update_job_status(self, 
                          job_id: str, 
                          status: str, 
                          message: str,
                          result_file: str = None,
                          error: str = None):
        """更新任务状态"""
        if job_id in self.job_status:
            self.job_status[job_id]["status"] = status
            self.job_status[job_id]["message"] = message
            
            if status in ["completed", "failed"]:
                self.job_status[job_id]["completed_at"] = datetime.now().isoformat()
            
            if result_file:
                self.job_status[job_id]["result_file"] = result_file
            
            if error:
                self.job_status[job_id]["error"] = error
    
    def get_job_status(self, job_id: str) -> Optional[dict]:
        """获取任务状态"""
        return self.job_status.get(job_id)
    
    def cleanup_old_jobs(self, max_age_hours: int = 24):
        """清理旧任务"""
        now = datetime.now()
        to_remove = []
        
        for job_id, status in self.job_status.items():
            created_at = datetime.fromisoformat(status["created_at"])
            age = (now - created_at).total_seconds() / 3600
            
            if age > max_age_hours:
                # 删除任务目录
                job_dir = self.jobs_dir / job_id
                if job_dir.exists():
                    shutil.rmtree(job_dir)
                to_remove.append(job_id)
        
        for job_id in to_remove:
            del self.job_status[job_id]
        
        logger.info(f"清理了 {len(to_remove)} 个旧任务")


# ==================== FastAPI应用 ====================

# 创建FastAPI应用
app = FastAPI(
    title="PRT特征识别API服务",
    description="提供PRT/STEP文件的机械特征识别和提取服务",
    version="2.0.0",
    docs_url="/docs",
    redoc_url="/redoc"
)

# 添加CORS中间件
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# 创建特征提取服务实例
extractor_service = FeatureExtractorService()

# 全局变量：主板验证状态
MOTHERBOARD_VERIFIED = False


# ==================== 启动事件 ====================

@app.on_event("startup")
async def startup_event():
    """应用启动时的初始化"""
    global MOTHERBOARD_VERIFIED
    
    logger.info("="*60)
    logger.info("PRT特征识别API服务启动中...")
    logger.info("="*60)
    
    # 验证主板UID
    MOTHERBOARD_VERIFIED = verify_motherboard_uid("024AFQCNL8002172")
    
    if not MOTHERBOARD_VERIFIED:
        logger.warning("主板UID验证失败！服务将以受限模式运行")
    else:
        logger.info("主板UID验证通过")
    
    logger.info(f"临时目录: {extractor_service.temp_dir}")
    logger.info(f"任务目录: {extractor_service.jobs_dir}")
    logger.info("="*60)


@app.on_event("shutdown")
async def shutdown_event():
    """应用关闭时的清理"""
    logger.info("正在关闭服务并清理资源...")


# ==================== API端点 ====================

@app.get("/", response_model=dict)
async def root():
    """根路径"""
    return {
        "service": "PRT特征识别API服务",
        "version": "2.0.0",
        "status": "running",
        "docs": "/docs",
        "health": "/health"
    }


@app.get("/health", response_model=HealthResponse)
async def health_check():
    """健康检查"""
    return HealthResponse(
        status="healthy" if MOTHERBOARD_VERIFIED else "limited",
        version="2.0.0",
        motherboard_uid_verified=MOTHERBOARD_VERIFIED,
        timestamp=datetime.now().isoformat()
    )


@app.post("/api/v1/extract/upload", response_model=JobStatus)
async def upload_and_extract(
    background_tasks: BackgroundTasks,
    file: UploadFile = File(...),
    enable_topology_relations: bool = Query(True, description="启用拓扑关系分析"),
    enable_process_planning: bool = Query(False, description="启用工艺路线规划")
):
    """
    上传PRT/STEP文件并提取特征（异步）
    
    - **file**: 上传的PRT或STEP文件
    - **enable_topology_relations**: 是否启用拓扑关系分析
    - **enable_process_planning**: 是否启用工艺路线规划
    """
    # 检查主板验证
    if not MOTHERBOARD_VERIFIED:
        raise HTTPException(
            status_code=403,
            detail="服务未授权：主板UID验证失败"
        )
    
    # 检查文件类型
    if not file.filename.lower().endswith(('.stp', '.step', '.prt')):
        raise HTTPException(
            status_code=400,
            detail="不支持的文件类型，仅支持.stp, .step, .prt文件"
        )
    
    # 创建任务
    job_id = extractor_service.create_job()
    
    # 保存上传的文件
    job_dir = extractor_service.jobs_dir / job_id
    job_dir.mkdir(exist_ok=True)
    
    file_path = job_dir / file.filename
    
    try:
        # 保存文件
        with open(file_path, "wb") as buffer:
            content = await file.read()
            buffer.write(content)
        
        logger.info(f"文件已保存: {file_path}, 大小: {len(content)} bytes")
        
        # 添加后台任务
        background_tasks.add_task(
            extractor_service.process_file,
            job_id,
            file_path,
            enable_topology_relations,
            enable_process_planning
        )
        
        return JobStatus(**extractor_service.get_job_status(job_id))
        
    except Exception as e:
        logger.error(f"文件上传失败: {str(e)}")
        extractor_service.update_job_status(
            job_id, 
            "failed", 
            "文件上传失败",
            error=str(e)
        )
        raise HTTPException(status_code=500, detail=f"文件上传失败: {str(e)}")


@app.get("/api/v1/jobs/{job_id}", response_model=JobStatus)
async def get_job_status(job_id: str):
    """
    获取任务状态
    
    - **job_id**: 任务ID
    """
    status = extractor_service.get_job_status(job_id)
    
    if not status:
        raise HTTPException(status_code=404, detail="任务不存在")
    
    return JobStatus(**status)


@app.get("/api/v1/jobs/{job_id}/result")
async def get_job_result(job_id: str):
    """
    获取任务结果文件
    
    - **job_id**: 任务ID
    """
    status = extractor_service.get_job_status(job_id)
    
    if not status:
        raise HTTPException(status_code=404, detail="任务不存在")
    
    if status["status"] != "completed":
        raise HTTPException(
            status_code=400, 
            detail=f"任务未完成，当前状态: {status['status']}"
        )
    
    if not status["result_file"]:
        raise HTTPException(status_code=404, detail="结果文件不存在")
    
    result_file = extractor_service.jobs_dir / status["result_file"]
    
    if not result_file.exists():
        raise HTTPException(status_code=404, detail="结果文件已被清理")
    
    return FileResponse(
        result_file,
        media_type="application/json",
        filename=result_file.name
    )


@app.get("/api/v1/jobs")
async def list_jobs():
    """
    列出所有任务
    """
    jobs = [JobStatus(**status) for status in extractor_service.job_status.values()]
    return {
        "total": len(jobs),
        "jobs": jobs
    }


@app.delete("/api/v1/jobs/{job_id}")
async def delete_job(job_id: str):
    """
    删除任务及其结果
    
    - **job_id**: 任务ID
    """
    status = extractor_service.get_job_status(job_id)
    
    if not status:
        raise HTTPException(status_code=404, detail="任务不存在")
    
    # 删除任务目录
    job_dir = extractor_service.jobs_dir / job_id
    if job_dir.exists():
        shutil.rmtree(job_dir)
    
    # 删除状态记录
    del extractor_service.job_status[job_id]
    
    return {"message": "任务已删除", "job_id": job_id}


@app.post("/api/v1/cleanup")
async def cleanup_old_jobs(max_age_hours: int = Query(24, description="最大保留时间（小时）")):
    """
    清理旧任务
    
    - **max_age_hours**: 超过此时间的任务将被清理
    """
    if not MOTHERBOARD_VERIFIED:
        raise HTTPException(
            status_code=403,
            detail="服务未授权：主板UID验证失败"
        )
    
    extractor_service.cleanup_old_jobs(max_age_hours)
    return {"message": f"已清理超过 {max_age_hours} 小时的旧任务"}


# ==================== 主函数 ====================

def main():
    """启动API服务"""
    import argparse
    
    parser = argparse.ArgumentParser(description="PRT特征识别API服务")
    parser.add_argument("--host", type=str, default="0.0.0.0", help="服务地址")
    parser.add_argument("--port", type=int, default=8000, help="服务端口")
    parser.add_argument("--reload", action="store_true", help="启用热重载")
    parser.add_argument("--workers", type=int, default=1, help="工作进程数")
    
    args = parser.parse_args()
    
    uvicorn.run(
        "prt_feature_api:app",
        host=args.host,
        port=args.port,
        reload=args.reload,
        workers=args.workers
    )


if __name__ == "__main__":
    main()
