# API接口文档

## 基础信息

- **基础URL**: `http://your-server:8000`
- **API版本**: v1
- **数据格式**: JSON
- **字符编码**: UTF-8

## 认证方式

当前版本基于主板UID验证，未来版本将支持：
- API Key认证
- JWT Token认证
- OAuth 2.0

## 通用响应格式

### 成功响应
```json
{
  "data": {},
  "message": "操作成功"
}
```

### 错误响应
```json
{
  "detail": "错误描述"
}
```

## HTTP状态码

| 状态码 | 含义 | 说明 |
|--------|------|------|
| 200 | OK | 请求成功 |
| 201 | Created | 资源创建成功 |
| 400 | Bad Request | 请求参数错误 |
| 401 | Unauthorized | 未授权 |
| 403 | Forbidden | 禁止访问（主板验证失败） |
| 404 | Not Found | 资源不存在 |
| 500 | Internal Server Error | 服务器内部错误 |

---

## API端点详情

### 1. 根路径

#### GET /

获取服务基本信息

**请求示例**
```bash
curl http://localhost:8000/
```

**响应示例**
```json
{
  "service": "PRT特征识别API服务",
  "version": "2.0.0",
  "status": "running",
  "docs": "/docs",
  "health": "/health"
}
```

---

### 2. 健康检查

#### GET /health

检查服务健康状态

**请求示例**
```bash
curl http://localhost:8000/health
```

**响应示例**
```json
{
  "status": "healthy",
  "version": "2.0.0",
  "motherboard_uid_verified": true,
  "timestamp": "2025-11-18T10:30:00"
}
```

**字段说明**
- `status`: 服务状态（healthy/limited）
- `version`: API版本
- `motherboard_uid_verified`: 主板验证状态
- `timestamp`: 检查时间戳

---

### 3. 上传文件并提取特征

#### POST /api/v1/extract/upload

上传PRT/STEP文件进行特征提取（异步处理）

**请求参数**

| 参数名 | 类型 | 必填 | 说明 |
|--------|------|------|------|
| file | File | 是 | PRT/STEP文件 |
| enable_topology_relations | boolean | 否 | 启用拓扑关系分析（默认true） |
| enable_process_planning | boolean | 否 | 启用工艺规划（默认false） |

**请求示例**
```bash
curl -X POST "http://localhost:8000/api/v1/extract/upload" \
  -F "file=@model.stp" \
  -F "enable_topology_relations=true" \
  -F "enable_process_planning=false"
```

**响应示例**
```json
{
  "job_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
  "status": "pending",
  "message": "任务已创建",
  "created_at": "2025-11-18T10:30:00",
  "completed_at": null,
  "result_file": null,
  "error": null
}
```

**任务状态说明**
- `pending`: 等待处理
- `processing`: 正在处理
- `completed`: 处理完成
- `failed`: 处理失败

**错误响应**
```json
{
  "detail": "不支持的文件类型，仅支持.stp, .step, .prt文件"
}
```

---

### 4. 查询任务状态

#### GET /api/v1/jobs/{job_id}

查询特定任务的当前状态

**路径参数**
- `job_id`: 任务ID（UUID格式）

**请求示例**
```bash
curl http://localhost:8000/api/v1/jobs/a1b2c3d4-e5f6-7890-abcd-ef1234567890
```

**响应示例（处理中）**
```json
{
  "job_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
  "status": "processing",
  "message": "正在提取特征...",
  "created_at": "2025-11-18T10:30:00",
  "completed_at": null,
  "result_file": null,
  "error": null
}
```

**响应示例（已完成）**
```json
{
  "job_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
  "status": "completed",
  "message": "特征提取完成",
  "created_at": "2025-11-18T10:30:00",
  "completed_at": "2025-11-18T10:31:30",
  "result_file": "a1b2c3d4-e5f6-7890-abcd-ef1234567890/model_features.json",
  "error": null
}
```

**响应示例（失败）**
```json
{
  "job_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890",
  "status": "failed",
  "message": "处理失败",
  "created_at": "2025-11-18T10:30:00",
  "completed_at": "2025-11-18T10:30:30",
  "result_file": null,
  "error": "无法读取STEP文件: ..."
}
```

---

### 5. 获取任务结果

#### GET /api/v1/jobs/{job_id}/result

下载任务的特征提取结果文件

**路径参数**
- `job_id`: 任务ID（UUID格式）

**请求示例**
```bash
curl http://localhost:8000/api/v1/jobs/a1b2c3d4-e5f6-7890-abcd-ef1234567890/result \
  -o result.json
```

**响应类型**: `application/json`

**响应示例**（完整JSON，见下方结果格式部分）

**错误响应**
```json
{
  "detail": "任务未完成，当前状态: processing"
}
```

---

### 6. 列出所有任务

#### GET /api/v1/jobs

获取所有任务的列表及其状态

**请求示例**
```bash
curl http://localhost:8000/api/v1/jobs
```

**响应示例**
```json
{
  "total": 3,
  "jobs": [
    {
      "job_id": "job-id-1",
      "status": "completed",
      "message": "特征提取完成",
      "created_at": "2025-11-18T10:30:00",
      "completed_at": "2025-11-18T10:31:30",
      "result_file": "job-id-1/model_features.json",
      "error": null
    },
    {
      "job_id": "job-id-2",
      "status": "processing",
      "message": "正在提取特征...",
      "created_at": "2025-11-18T10:35:00",
      "completed_at": null,
      "result_file": null,
      "error": null
    },
    {
      "job_id": "job-id-3",
      "status": "pending",
      "message": "任务已创建",
      "created_at": "2025-11-18T10:40:00",
      "completed_at": null,
      "result_file": null,
      "error": null
    }
  ]
}
```

---

### 7. 删除任务

#### DELETE /api/v1/jobs/{job_id}

删除指定任务及其结果文件

**路径参数**
- `job_id`: 任务ID（UUID格式）

**请求示例**
```bash
curl -X DELETE http://localhost:8000/api/v1/jobs/a1b2c3d4-e5f6-7890-abcd-ef1234567890
```

**响应示例**
```json
{
  "message": "任务已删除",
  "job_id": "a1b2c3d4-e5f6-7890-abcd-ef1234567890"
}
```

---

### 8. 清理旧任务

#### POST /api/v1/cleanup

清理超过指定时间的旧任务

**查询参数**
- `max_age_hours`: 最大保留时间（小时），默认24

**请求示例**
```bash
curl -X POST "http://localhost:8000/api/v1/cleanup?max_age_hours=24"
```

**响应示例**
```json
{
  "message": "已清理超过 24 小时的旧任务"
}
```

---

## 特征提取结果格式

### JSON结构概览

```json
{
  "metadata": {
    "extraction_time": "2025-11-18T10:31:30",
    "version": "2.0.0-api",
    "filename": "model.stp"
  },
  "advanced_features": {
    "holes": [...],
    "pockets": [...],
    "fillets": [...],
    "chamfers": [...]
  },
  "basic_features": {
    "faces": [...],
    "edges": [...],
    "vertices": [...]
  },
  "topology": {
    "feature_adjacency": [...],
    "feature_groups": [...]
  }
}
```

### 孔特征示例

```json
{
  "id": "hole_001",
  "type": "through_hole",
  "diameter": 10.0,
  "depth": 50.0,
  "position": [0.0, 0.0, 0.0],
  "direction": [0.0, 0.0, 1.0],
  "axis_vector": [0.0, 0.0, 1.0],
  "is_blind": false,
  "is_threaded": false
}
```

**孔类型**
- `through_hole`: 通孔
- `blind_hole`: 盲孔
- `countersunk_hole`: 沉头孔
- `counterbored_hole`: 锪孔

### 槽特征示例

```json
{
  "id": "pocket_001",
  "type": "rectangular_pocket",
  "width": 20.0,
  "length": 50.0,
  "depth": 10.0,
  "position": [10.0, 10.0, 0.0],
  "orientation": [0.0, 0.0, 1.0],
  "corner_radius": 2.0
}
```

**槽类型**
- `rectangular_pocket`: 矩形槽
- `t_slot`: T型槽
- `dovetail_slot`: 燕尾槽

### 圆角特征示例

```json
{
  "id": "fillet_001",
  "type": "constant_radius_fillet",
  "radius": 5.0,
  "edges": [
    {
      "edge_id": "edge_123",
      "length": 100.0
    }
  ],
  "adjacent_faces": ["face_001", "face_002"]
}
```

### 拓扑关系示例

```json
{
  "feature_adjacency": [
    {
      "feature1_id": "hole_001",
      "feature2_id": "pocket_001",
      "relation_type": "shared_face",
      "shared_entity": "face_025"
    }
  ]
}
```

---

## 使用流程示例

### 完整工作流

```python
import requests
import time

BASE_URL = "http://localhost:8000"

# 1. 健康检查
response = requests.get(f"{BASE_URL}/health")
print(response.json())

# 2. 上传文件
with open("model.stp", "rb") as f:
    files = {"file": ("model.stp", f)}
    params = {
        "enable_topology_relations": True,
        "enable_process_planning": False
    }
    response = requests.post(
        f"{BASE_URL}/api/v1/extract/upload",
        files=files,
        params=params
    )
    job_info = response.json()
    job_id = job_info["job_id"]
    print(f"任务ID: {job_id}")

# 3. 轮询任务状态
while True:
    response = requests.get(f"{BASE_URL}/api/v1/jobs/{job_id}")
    status = response.json()
    print(f"状态: {status['status']}")
    
    if status["status"] == "completed":
        break
    elif status["status"] == "failed":
        print(f"错误: {status['error']}")
        break
    
    time.sleep(2)

# 4. 获取结果
response = requests.get(f"{BASE_URL}/api/v1/jobs/{job_id}/result")
result = response.json()
print(f"提取到 {len(result['advanced_features'])} 个特征类型")

# 5. 清理任务（可选）
response = requests.delete(f"{BASE_URL}/api/v1/jobs/{job_id}")
print(response.json())
```

---

## 性能指标

| 操作 | 预期时间 |
|------|----------|
| API响应时间 | < 100ms |
| 小文件处理（<10MB） | < 30秒 |
| 大文件处理（100-500MB） | < 5分钟 |
| 并发处理能力 | 10+ QPS |

---

## 限制说明

| 项目 | 限制 |
|------|------|
| 最大文件大小 | 500MB |
| 单次上传文件数 | 1个 |
| 并发任务数 | 无限制（受服务器资源限制） |
| 任务保留时间 | 24小时（可配置） |
| API调用频率 | 无限制（可配置限流） |

---

## 错误处理

### 常见错误代码

| 错误信息 | 原因 | 解决方案 |
|----------|------|----------|
| "不支持的文件类型" | 文件扩展名不是.stp/.step/.prt | 使用正确格式的文件 |
| "任务不存在" | job_id无效或已过期 | 检查job_id是否正确 |
| "任务未完成" | 尝试获取未完成任务的结果 | 等待任务完成后再获取结果 |
| "服务未授权" | 主板UID验证失败 | 在授权计算机上运行服务 |
| "文件上传失败" | 网络或服务器问题 | 检查网络连接，重试 |

---

## 版本历史

### v2.0.0 (2025-11-18)
- ✨ 初始API版本发布
- ✨ 支持异步任务处理
- ✨ 完整的RESTful接口
- ✨ 自动生成API文档

---

## 技术支持

遇到问题？
1. 查看服务日志
2. 检查 `/health` 端点
3. 阅读完整文档
4. 联系技术支持
