"""
三维模型智能搜索系统 - 完整测试脚本
测试LLM增强功能
"""

import os
import sys
import json

# 添加模块路径
sys.path.insert(0, '/home/claude/cad_semantic_search')

from feature_semantic_analyzer import FeatureSemanticAnalyzer
from llm_enhanced import LLMConfig, LLMEnhancedAnalyzer, create_llm_provider
from llm_search_engine import LLMEnhancedSearchEngine, create_enhanced_search_engine


def print_section(title):
    print("\n" + "=" * 60)
    print(f"  {title}")
    print("=" * 60)


def test_basic_analysis():
    """测试基础分析功能"""
    print_section("1. 基础特征分析（规则引擎）")
    
    analyzer = FeatureSemanticAnalyzer()
    
    test_files = [
        "/mnt/user-data/uploads/AGES-50_features.json",
        "/mnt/user-data/uploads/凹端内六角紧定螺钉_PSEP-6-16-A__features.json",
        "/mnt/user-data/uploads/内六角喉塞_PSEG-M22-A__features.json",
    ]
    
    for filepath in test_files:
        try:
            result = analyzer.analyze_json(filepath)
            print(f"\n📦 {result.filename}")
            print(f"   类型: {result.part_type}")
            print(f"   关键词: {', '.join(result.keywords[:8])}")
            print(f"   描述: {result.description[:80]}...")
        except Exception as e:
            print(f"✗ 分析失败: {filepath} - {e}")


def test_llm_availability():
    """测试LLM可用性"""
    print_section("2. LLM可用性检测")
    
    # 检查环境变量
    api_key = os.getenv("ANTHROPIC_API_KEY", "")
    
    if api_key:
        print(f"✓ ANTHROPIC_API_KEY 已设置 (长度: {len(api_key)})")
    else:
        print("⚠ ANTHROPIC_API_KEY 未设置")
        print("  设置方法: export ANTHROPIC_API_KEY=your_key")
    
    # 测试各提供者
    providers = [
        ("anthropic", LLMConfig(provider="anthropic", api_key=api_key)),
        ("openai", LLMConfig(provider="openai", api_key=os.getenv("OPENAI_API_KEY", ""))),
        ("ollama", LLMConfig(provider="ollama", base_url="http://localhost:11434")),
    ]
    
    print("\n提供者状态:")
    for name, config in providers:
        provider = create_llm_provider(config)
        status = "✓ 可用" if provider.is_available() else "✗ 不可用"
        print(f"  {name}: {status}")
    
    return bool(api_key)


def test_llm_enhanced_analysis(api_key: str):
    """测试LLM增强分析"""
    print_section("3. LLM增强分析")
    
    config = LLMConfig(
        provider="anthropic",
        model="claude-sonnet-4-20250514",
        api_key=api_key,
    )
    
    analyzer = LLMEnhancedAnalyzer(config)
    
    if not analyzer.provider.is_available():
        print("⚠ LLM不可用，跳过此测试")
        return
    
    # 加载测试数据
    test_file = "/mnt/user-data/uploads/AGES-50_features.json"
    with open(test_file, 'r', encoding='utf-8') as f:
        features_data = json.load(f)
    
    print(f"\n分析文件: {test_file}")
    print("正在调用LLM...")
    
    result = analyzer.analyze_part(features_data)
    
    if result:
        print(f"\n✓ LLM分析结果:")
        print(f"  零件类型: {result.get('part_type', 'N/A')}")
        print(f"  零件大类: {result.get('part_category', 'N/A')}")
        print(f"  功能: {result.get('function', 'N/A')}")
        print(f"  描述: {result.get('description', 'N/A')[:100]}...")
        print(f"  关键词: {', '.join(result.get('keywords', [])[:10])}")
        print(f"  行业术语: {', '.join(result.get('industry_terms', [])[:5])}")
        print(f"  相似零件: {', '.join(result.get('similar_parts', [])[:5])}")
        print(f"  推荐材料: {result.get('material_suggestion', 'N/A')}")
        print(f"  相关标准: {', '.join(result.get('standards', [])[:3])}")
    else:
        print("✗ LLM分析失败")


def test_query_understanding(api_key: str):
    """测试查询意图理解"""
    print_section("4. 查询意图理解")
    
    config = LLMConfig(
        provider="anthropic",
        api_key=api_key,
    )
    
    analyzer = LLMEnhancedAnalyzer(config)
    
    if not analyzer.provider.is_available():
        print("⚠ LLM不可用，跳过此测试")
        return
    
    test_queries = [
        "三通",
        "M6内六角螺钉",
        "DN50法兰盘",
        "不锈钢管接头",
    ]
    
    for query in test_queries:
        print(f"\n🔍 查询: '{query}'")
        result = analyzer.understand_query(query)
        
        if result:
            print(f"   意图: {result.get('intent', 'N/A')}")
            print(f"   零件类型: {result.get('part_types', [])}")
            print(f"   规格: {result.get('specifications', [])}")
            print(f"   扩展词: {result.get('expanded_queries', [])[:5]}")
            print(f"   同义词: {result.get('synonyms', [])[:5]}")


def test_smart_search(api_key: str):
    """测试智能搜索"""
    print_section("5. 智能搜索测试")
    
    # 创建LLM配置
    llm_config = LLMConfig(
        provider="anthropic",
        api_key=api_key,
    )
    
    # 测试文件
    test_files = [
        "/mnt/user-data/uploads/AGES-50_features.json",
        "/mnt/user-data/uploads/prt0001_features.json",
        "/mnt/user-data/uploads/凹端内六角紧定螺钉_PSEP-6-16-A__features.json",
        "/mnt/user-data/uploads/内六角喉塞_PSEG-M22-A__features.json",
        "/mnt/user-data/uploads/内六角机螺钉_内六角平头带垫螺栓_PSHFP-M6-25-A__features.json",
    ]
    
    # 创建引擎
    print("\n构建LLM增强索引...")
    engine = create_enhanced_search_engine(test_files, llm_config)
    
    # 测试搜索
    test_queries = ["三通管件", "螺丝", "堵头", "紧固件"]
    
    for query in test_queries:
        print(f"\n🔍 搜索: '{query}'")
        print("-" * 40)
        
        results = engine.smart_search(query, top_k=3, explain=True)
        
        if not results:
            print("  无结果")
            continue
        
        for i, r in enumerate(results, 1):
            print(f"  {i}. {r.filename}")
            print(f"     类型: {r.part_type} | 分数: {r.score:.3f}")
            if r.llm_explanation:
                print(f"     💡 {r.llm_explanation}")
            if r.related_parts:
                print(f"     相关: {', '.join(r.related_parts)}")


def test_conversational_search(api_key: str):
    """测试对话式搜索"""
    print_section("6. 对话式搜索测试")
    
    llm_config = LLMConfig(
        provider="anthropic",
        api_key=api_key,
    )
    
    test_files = [
        "/mnt/user-data/uploads/AGES-50_features.json",
        "/mnt/user-data/uploads/内六角喉塞_PSEG-M22-A__features.json",
    ]
    
    engine = create_enhanced_search_engine(test_files, llm_config)
    
    if not engine.llm_available:
        print("⚠ LLM不可用，跳过此测试")
        return
    
    # 模拟对话
    conversation = [
        "我需要找一个管道连接件",
        "有没有三通的？",
        "这个是什么材料的？",
    ]
    
    history = []
    
    for message in conversation:
        print(f"\n👤 用户: {message}")
        
        result = engine.conversational_search(message, history)
        
        print(f"🤖 助手: {result['message']}")
        
        if result['suggestions']:
            print(f"   建议问题: {', '.join(result['suggestions'][:2])}")
        
        # 更新历史
        history.append({"role": "user", "content": message})
        history.append({"role": "assistant", "content": result['message']})


def main():
    """主测试流程"""
    print("\n" + "🔧" * 30)
    print("  三维模型智能搜索系统 - 完整测试")
    print("🔧" * 30)
    
    # 1. 基础分析
    test_basic_analysis()
    
    # 2. LLM可用性
    has_api_key = test_llm_availability()
    
    api_key = os.getenv("ANTHROPIC_API_KEY", "")
    
    if has_api_key:
        # 3. LLM增强分析
        test_llm_enhanced_analysis(api_key)
        
        # 4. 查询理解
        test_query_understanding(api_key)
        
        # 5. 智能搜索
        test_smart_search(api_key)
        
        # 6. 对话式搜索
        test_conversational_search(api_key)
    else:
        print("\n" + "=" * 60)
        print("⚠ 未检测到API Key，LLM增强功能测试已跳过")
        print("=" * 60)
        print("\n要启用LLM增强功能，请设置环境变量:")
        print("  export ANTHROPIC_API_KEY=your_anthropic_key")
        print("  # 或")
        print("  export OPENAI_API_KEY=your_openai_key")
        print("\n系统将自动回退到规则引擎模式运行。")
    
    print("\n" + "=" * 60)
    print("✅ 测试完成")
    print("=" * 60)


if __name__ == "__main__":
    main()
