"""
三维模型语义搜索 API 服务
FastAPI实现，支持与Dify等系统集成
"""

from fastapi import FastAPI, HTTPException, Query, UploadFile, File
from fastapi.middleware.cors import CORSMiddleware
from pydantic import BaseModel, Field
from typing import List, Optional, Dict, Any
import json
import os
from pathlib import Path
import tempfile
import uvicorn

# 导入搜索引擎模块
from feature_semantic_analyzer import FeatureSemanticAnalyzer
from search_engine import CADModelSearchEngine, SearchResult

app = FastAPI(
    title="三维模型语义搜索API",
    description="基于特征的CAD模型智能检索系统",
    version="1.0.0"
)

# CORS配置
app.add_middleware(
    CORSMiddleware,
    allow_origins=["*"],
    allow_credentials=True,
    allow_methods=["*"],
    allow_headers=["*"],
)

# 全局实例
analyzer = FeatureSemanticAnalyzer()
search_engine = CADModelSearchEngine()

# 数据目录
DATA_DIR = Path("./data")
INDEX_FILE = DATA_DIR / "search_index.json"


# ============== Pydantic Models ==============

class SearchRequest(BaseModel):
    query: str = Field(..., description="搜索查询词", example="三通")
    top_k: int = Field(default=10, ge=1, le=100, description="返回结果数量")
    method: str = Field(default="hybrid", description="搜索方法: keyword, semantic, hybrid")


class SearchResultItem(BaseModel):
    file_id: str
    filename: str
    part_type: str
    score: float
    match_type: str
    highlights: List[str]
    description: str


class SearchResponse(BaseModel):
    query: str
    total_results: int
    results: List[SearchResultItem]


class ModelInfo(BaseModel):
    file_id: str
    filename: str
    part_type: str
    keywords: List[str]
    description: str
    feature_summary: Dict[str, Any]


class AnalyzeResponse(BaseModel):
    success: bool
    file_id: str
    filename: str
    part_type: str
    keywords: List[str]
    description: str
    message: str


class IndexStats(BaseModel):
    total_models: int
    part_types: Dict[str, int]
    total_keywords: int


# ============== API Endpoints ==============

@app.get("/", tags=["健康检查"])
async def root():
    """API健康检查"""
    return {
        "service": "三维模型语义搜索API",
        "status": "running",
        "version": "1.0.0"
    }


@app.post("/search", response_model=SearchResponse, tags=["搜索"])
async def search_models(request: SearchRequest):
    """
    搜索三维模型
    
    支持以下搜索方法:
    - keyword: 基于关键词的精确匹配
    - semantic: 基于语义向量的相似度搜索
    - hybrid: 混合搜索（推荐）
    
    示例查询:
    - "三通" - 搜索三通管件
    - "内六角螺钉" - 搜索内六角螺钉
    - "M6" - 搜索M6规格的零件
    """
    results = search_engine.search(
        query=request.query,
        top_k=request.top_k,
        method=request.method
    )
    
    return SearchResponse(
        query=request.query,
        total_results=len(results),
        results=[
            SearchResultItem(
                file_id=r.file_id,
                filename=r.filename,
                part_type=r.part_type,
                score=r.score,
                match_type=r.match_type,
                highlights=r.highlights,
                description=r.description
            ) for r in results
        ]
    )


@app.get("/search", response_model=SearchResponse, tags=["搜索"])
async def search_models_get(
    query: str = Query(..., description="搜索查询词"),
    top_k: int = Query(default=10, ge=1, le=100),
    method: str = Query(default="hybrid")
):
    """GET方式搜索（方便浏览器测试）"""
    return await search_models(SearchRequest(query=query, top_k=top_k, method=method))


@app.post("/analyze", response_model=AnalyzeResponse, tags=["分析"])
async def analyze_json_file(file: UploadFile = File(...)):
    """
    上传并分析JSON特征文件
    
    分析完成后自动添加到搜索索引
    """
    if not file.filename.endswith('.json'):
        raise HTTPException(status_code=400, detail="只支持JSON文件")
    
    try:
        # 保存临时文件
        content = await file.read()
        
        # 确保数据目录存在
        DATA_DIR.mkdir(parents=True, exist_ok=True)
        
        # 保存到数据目录
        save_path = DATA_DIR / file.filename
        with open(save_path, 'wb') as f:
            f.write(content)
        
        # 分析文件
        semantics = analyzer.analyze_json(str(save_path))
        semantic_dict = analyzer.to_dict(semantics)
        
        # 添加到索引
        search_engine.index_model(semantic_dict)
        search_engine.build_vector_index()
        
        # 保存索引
        search_engine.save_index(str(INDEX_FILE))
        
        return AnalyzeResponse(
            success=True,
            file_id=semantics.file_id,
            filename=semantics.filename,
            part_type=semantics.part_type,
            keywords=semantics.keywords,
            description=semantics.description,
            message="文件分析并索引成功"
        )
        
    except json.JSONDecodeError:
        raise HTTPException(status_code=400, detail="无效的JSON格式")
    except Exception as e:
        raise HTTPException(status_code=500, detail=f"分析失败: {str(e)}")


@app.get("/models", response_model=List[ModelInfo], tags=["模型管理"])
async def list_models(
    part_type: Optional[str] = Query(default=None, description="按零件类型过滤")
):
    """列出所有已索引的模型"""
    results = []
    
    for file_id, data in search_engine.models.items():
        if part_type and data.get("part_type") != part_type:
            continue
        
        results.append(ModelInfo(
            file_id=file_id,
            filename=data.get("filename", ""),
            part_type=data.get("part_type", ""),
            keywords=data.get("keywords", []),
            description=data.get("description", ""),
            feature_summary=data.get("feature_summary", {})
        ))
    
    return results


@app.get("/models/{file_id}", response_model=ModelInfo, tags=["模型管理"])
async def get_model(file_id: str):
    """获取单个模型的详细信息"""
    if file_id not in search_engine.models:
        raise HTTPException(status_code=404, detail="模型不存在")
    
    data = search_engine.models[file_id]
    return ModelInfo(
        file_id=file_id,
        filename=data.get("filename", ""),
        part_type=data.get("part_type", ""),
        keywords=data.get("keywords", []),
        description=data.get("description", ""),
        feature_summary=data.get("feature_summary", {})
    )


@app.delete("/models/{file_id}", tags=["模型管理"])
async def delete_model(file_id: str):
    """从索引中删除模型"""
    if file_id not in search_engine.models:
        raise HTTPException(status_code=404, detail="模型不存在")
    
    # 从各索引中删除
    del search_engine.models[file_id]
    if file_id in search_engine.vectors:
        del search_engine.vectors[file_id]
    
    # 重建关键词索引
    for keyword, ids in list(search_engine.keyword_index.items()):
        ids.discard(file_id)
        if not ids:
            del search_engine.keyword_index[keyword]
    
    # 重建类型索引
    for part_type, ids in list(search_engine.part_type_index.items()):
        ids.discard(file_id)
        if not ids:
            del search_engine.part_type_index[part_type]
    
    # 保存更新后的索引
    search_engine.save_index(str(INDEX_FILE))
    
    return {"message": f"模型 {file_id} 已删除"}


@app.get("/stats", response_model=IndexStats, tags=["统计"])
async def get_stats():
    """获取索引统计信息"""
    part_type_counts = {}
    for part_type, ids in search_engine.part_type_index.items():
        part_type_counts[part_type] = len(ids)
    
    return IndexStats(
        total_models=len(search_engine.models),
        part_types=part_type_counts,
        total_keywords=len(search_engine.keyword_index)
    )


@app.post("/rebuild-index", tags=["管理"])
async def rebuild_index():
    """重建搜索索引"""
    global search_engine
    
    # 创建新引擎
    new_engine = CADModelSearchEngine()
    
    # 重新索引所有数据目录中的文件
    if DATA_DIR.exists():
        for json_file in DATA_DIR.glob("*_features.json"):
            try:
                semantics = analyzer.analyze_json(str(json_file))
                semantic_dict = analyzer.to_dict(semantics)
                new_engine.index_model(semantic_dict)
            except Exception as e:
                print(f"索引失败: {json_file} - {e}")
    
    new_engine.build_vector_index()
    new_engine.save_index(str(INDEX_FILE))
    
    search_engine = new_engine
    
    return {
        "message": "索引重建完成",
        "total_models": len(search_engine.models)
    }


@app.on_event("startup")
async def startup_event():
    """服务启动时加载索引"""
    DATA_DIR.mkdir(parents=True, exist_ok=True)
    
    if INDEX_FILE.exists():
        try:
            search_engine.load_index(str(INDEX_FILE))
            print(f"已加载索引，包含 {len(search_engine.models)} 个模型")
        except Exception as e:
            print(f"加载索引失败: {e}")


# ============== Dify 工具集成格式 ==============

@app.post("/dify/search", tags=["Dify集成"])
async def dify_search(
    query: str = Query(..., description="搜索关键词"),
    limit: int = Query(default=5, ge=1, le=20)
):
    """
    Dify工作流集成接口
    
    返回格式适配Dify工具调用
    """
    results = search_engine.search(query=query, top_k=limit, method="hybrid")
    
    # 格式化为Dify友好的输出
    output_lines = []
    for i, r in enumerate(results, 1):
        output_lines.append(
            f"{i}. **{r.filename}**\n"
            f"   - 类型: {r.part_type}\n"
            f"   - 相关度: {r.score:.1%}\n"
            f"   - 描述: {r.description[:100]}..."
        )
    
    return {
        "success": True,
        "query": query,
        "count": len(results),
        "formatted_output": "\n\n".join(output_lines) if output_lines else "未找到匹配的模型",
        "raw_results": [
            {
                "file_id": r.file_id,
                "filename": r.filename,
                "part_type": r.part_type,
                "score": r.score
            } for r in results
        ]
    }


# ============== 主入口 ==============

def main():
    """运行API服务"""
    uvicorn.run(
        "api_server:app",
        host="0.0.0.0",
        port=8000,
        reload=True
    )


if __name__ == "__main__":
    main()
