# 三维模型语义搜索系统 (LLM增强版)

## 系统概述

本系统实现了基于CAD特征JSON的三维模型文字搜索功能，支持自然语言查询如"三通"、"内六角螺钉"等。

**v2.0 新增LLM增强功能：**
- 🤖 智能零件类型识别和语义描述生成
- 🔍 用户查询意图理解
- 💬 对话式搜索交互
- 🔗 自动同义词和术语扩展

## 核心架构

```
┌────────────────────────────────────────────────────────────────────────────┐
│                         三维模型语义搜索系统                                  │
├────────────────────────────────────────────────────────────────────────────┤
│                                                                            │
│   ┌─────────────┐        ┌─────────────────┐        ┌─────────────────┐   │
│   │  JSON特征   │───────▶│  语义分析器     │───────▶│  搜索索引       │   │
│   │  文件输入   │        │                 │        │                 │   │
│   └─────────────┘        │ • 特征提取      │        │ • 关键词索引    │   │
│                          │ • 拓扑分析      │        │ • 向量索引      │   │
│                          │ • 类型推断      │        │ • 类型索引      │   │
│                          │ • 描述生成      │        │                 │   │
│                          └─────────────────┘        └────────┬────────┘   │
│                                                              │            │
│   ┌──────────────────────────────────────────────────────────┼────────┐   │
│   │                        检索层                             │        │   │
│   │  ┌────────────────┐   ┌────────────────┐   ┌────────────▼───┐    │   │
│   │  │  关键词搜索    │   │  语义向量搜索  │   │  混合检索      │    │   │
│   │  │ (精确匹配)     │ + │ (相似度计算)   │ = │ (加权融合)     │    │   │
│   │  └────────────────┘   └────────────────┘   └────────────────┘    │   │
│   └──────────────────────────────────────────────────────────────────┘   │
│                                    │                                      │
│                                    ▼                                      │
│   ┌──────────────────────────────────────────────────────────────────┐   │
│   │                       输出接口                                    │   │
│   │  ┌─────────────┐   ┌─────────────┐   ┌─────────────────────┐    │   │
│   │  │ REST API    │   │ Dify集成    │   │ Neo4j知识图谱      │    │   │
│   │  │ (FastAPI)   │   │ 工具接口    │   │ 复杂拓扑查询       │    │   │
│   │  └─────────────┘   └─────────────┘   └─────────────────────┘    │   │
│   └──────────────────────────────────────────────────────────────────┘   │
│                                                                            │
└────────────────────────────────────────────────────────────────────────────┘
```

## 零件类型自动识别规则

| 零件类型 | 识别特征 | 示例 |
|---------|---------|------|
| 三通 | 3个不同轴向的交叉孔 + 凸台 | T型管接头 |
| 四通 | 4个不同轴向的交叉孔 + 凸台 | 十字管接头 |
| 螺钉 | 通孔 + 多个倒角 + 圆锥面 | 内六角螺钉 |
| 螺母 | 1个通孔 + 6-8个平面 | 六角螺母 |
| 喉塞 | 通孔 + 凸台 + 六角平面 | 堵头 |
| 法兰 | 多个均布孔 + 2个平面 | 连接法兰 |
| 弯头 | 2个不同轴向的孔 | 90°弯管 |

## 搜索功能

### 1. 同义词扩展
```python
synonyms = {
    "三通": ["T型", "T接头", "分流接头", "三向接头", "tee"],
    "四通": ["十字接头", "cross", "四向接头", "交叉接头"],
    "螺钉": ["螺栓", "螺丝", "screw", "bolt"],
    ...
}
```

### 2. 搜索方法
- **keyword**: 关键词精确匹配
- **semantic**: TF-IDF向量相似度
- **hybrid**: 加权混合（默认 60% 关键词 + 40% 语义）

### 3. 搜索示例
```python
# 搜索三通管件
results = engine.search("三通", top_k=10)

# 搜索内六角零件
results = engine.search("内六角", method="hybrid")

# 按特征类型搜索
results = engine.search("交叉孔")
```

## API 接口

### 基础搜索接口
```bash
# GET 方式
curl "http://localhost:8000/search?query=三通&top_k=10"

# POST 方式
curl -X POST "http://localhost:8000/search" \
  -H "Content-Type: application/json" \
  -d '{"query": "三通", "top_k": 10, "explain": true}'
```

### 对话式搜索（LLM增强）
```bash
curl -X POST "http://localhost:8000/chat" \
  -H "Content-Type: application/json" \
  -d '{
    "message": "我需要找一个管道连接件",
    "history": []
  }'
```

### 查询意图理解（LLM增强）
```bash
curl "http://localhost:8000/query/understand?query=M6内六角螺钉"

# 返回示例:
# {
#   "intent": "search",
#   "part_types": ["螺钉", "紧固件"],
#   "specifications": ["M6"],
#   "expanded_queries": ["内六角螺钉", "六角螺丝", "allen screw"],
#   "synonyms": ["螺栓", "螺丝", "screw"]
# }
```

### 上传分析
```bash
curl -X POST "http://localhost:8000/analyze" \
  -F "file=@model_features.json"
```

### Dify工具集成
```bash
curl "http://localhost:8000/dify/smart_search?query=螺钉&limit=5"
```

## 文件结构

```
cad_semantic_search/
├── feature_semantic_analyzer.py  # 特征语义分析器（规则引擎）
├── search_engine.py              # 基础搜索引擎
├── llm_enhanced.py               # LLM增强模块
├── llm_search_engine.py          # LLM增强搜索引擎
├── api_server.py                 # 基础API服务
├── api_server_enhanced.py        # LLM增强API服务
├── neo4j_integration.py          # Neo4j知识图谱集成
├── test_full_system.py           # 完整测试脚本
├── data/                         # 数据存储目录
└── README.md                     # 本文档
```

## LLM增强功能详解

### 1. 智能零件分析
LLM会分析CAD特征并生成：
- **零件类型和大类**：如"三通" → "管件"
- **功能描述**：零件的主要用途
- **行业术语**：专业术语和标准编号
- **相似零件推荐**：可替代的零件类型

```python
# LLM分析输出示例
{
    "part_type": "四通",
    "part_category": "管件",
    "function": "用于管道系统中四个方向的流体分流和汇合",
    "industry_terms": ["cross fitting", "4-way", "十字接头"],
    "similar_parts": ["三通", "弯头", "直通接头"],
    "standards": ["GB/T 14383", "ASME B16.9"],
    "material_suggestion": "不锈钢304/316或碳钢"
}
```

### 2. 查询意图理解
自动识别用户搜索意图：

| 查询 | 识别结果 |
|-----|---------|
| "三通" | 搜索管件类零件 |
| "M6螺丝" | 搜索M6规格紧固件 |
| "DN50法兰" | 搜索DN50管道法兰 |
| "不锈钢接头" | 搜索不锈钢材质连接件 |

### 3. 对话式交互
支持多轮对话，理解上下文：

```
👤 用户: 我需要找一个管道连接件
🤖 助手: 您需要什么类型的连接件？比如三通、四通、弯头或法兰？

👤 用户: 三通的
🤖 助手: 为您找到以下三通管件：
   1. AGES-50（四通，但也可作为三通使用）
   建议：您是否需要特定规格？

👤 用户: DN50的
🤖 助手: 抱歉，当前库中没有DN50规格的三通...
```

### 4. 支持的LLM提供者

| 提供者 | 配置方式 | 说明 |
|-------|---------|------|
| Anthropic Claude | `ANTHROPIC_API_KEY` | 推荐，效果最佳 |
| OpenAI | `OPENAI_API_KEY` | GPT-4/GPT-4o |
| Ollama | `LLM_PROVIDER=ollama` | 本地部署，无需API |
| 兼容API | `LLM_BASE_URL` | 通义千问、文心等 |

## 快速开始

### 1. 安装依赖
```bash
pip install fastapi uvicorn numpy

# LLM增强功能（可选但推荐）
pip install anthropic  # 使用Claude
# 或
pip install openai     # 使用OpenAI/兼容API
```

### 2. 配置LLM（可选）
```bash
# 方式1: Anthropic Claude（推荐）
export ANTHROPIC_API_KEY=your_anthropic_key

# 方式2: OpenAI
export OPENAI_API_KEY=your_openai_key

# 方式3: 兼容API（如本地Ollama、通义千问等）
export LLM_PROVIDER=http
export LLM_BASE_URL=http://localhost:11434/v1
export LLM_API_KEY=your_key
```

### 3. 启动服务
```bash
cd cad_semantic_search

# 基础版（规则引擎）
python api_server.py

# LLM增强版（推荐）
python api_server_enhanced.py

# 服务运行在 http://localhost:8000
```

### 4. 测试搜索
```bash
# 命令行测试
python test_full_system.py

# 访问API文档
open http://localhost:8000/docs
```

## 生产环境增强建议

### 1. 使用专业向量嵌入
```python
# 替换SimpleTextVectorizer为sentence-transformers
from sentence_transformers import SentenceTransformer
model = SentenceTransformer('paraphrase-multilingual-MiniLM-L12-v2')
embeddings = model.encode(texts)
```

### 2. 使用向量数据库
- **Milvus**: 大规模向量检索
- **Qdrant**: 轻量级部署
- **Neo4j Vector Index**: 图+向量混合

### 3. LLM增强
```python
# 使用LLM生成更丰富的语义描述
prompt = f"""
分析以下CAD特征JSON，生成自然语言描述：
{feature_json}
请识别：零件类型、主要功能、典型用途、相似零件
"""
```

## Dify工作流集成

### 工具配置（智能搜索）
```yaml
name: cad_smart_search
description: 智能搜索三维CAD模型库，支持自然语言查询
parameters:
  query:
    type: string
    description: 搜索关键词，如"三通管件"、"M6螺钉"
    required: true
  limit:
    type: integer
    description: 返回结果数量
    default: 5
endpoint: http://your-server:8000/dify/smart_search
method: POST
```

### 工具配置（对话式搜索）
```yaml
name: cad_chat
description: 对话式零件搜索助手
parameters:
  message:
    type: string
    description: 用户消息
    required: true
  context:
    type: string
    description: 对话历史JSON
    default: "[]"
endpoint: http://your-server:8000/dify/chat
method: POST
```

### 工作流示例
```
用户输入 
    ↓
LLM节点（提取搜索意图）
    ↓
条件分支
    ├── 搜索意图 → 调用cad_smart_search → 格式化结果
    ├── 询问意图 → 调用cad_chat → 返回对话响应
    └── 其他 → 直接回复
```

## 扩展：Neo4j复杂查询

```cypher
// 查找所有管件（有交叉孔的零件）
MATCH (m:Model)-[:HAS_FEATURE]->(f:Feature)
WHERE f.raw_data CONTAINS '交叉孔'
RETURN m.filename, m.part_type, count(f) as cross_holes
ORDER BY cross_holes DESC

// 查找拓扑相似的零件
MATCH (m1:Model {file_id: 'AGES-50'})-[:HAS_FEATURE]->(f1:Feature)
MATCH (m2:Model)-[:HAS_FEATURE]->(f2:Feature)
WHERE m1 <> m2 AND f1.feature_type = f2.feature_type
WITH m2, count(*) as common_features
ORDER BY common_features DESC
RETURN m2.filename, common_features
LIMIT 5
```

## 测试结果示例

### 规则引擎模式
```
🔍 搜索: '三通'
  1. AGES-50.ccm
     类型: 四通 | 分数: 1.000 | 匹配: hybrid
     高亮: 关键词匹配: 接头

🔍 搜索: '内六角'
  1. 内六角机螺钉 内六角平头带垫螺栓.ccm
     类型: 螺钉 | 分数: 0.987
  2. 内六角喉塞.ccm
     类型: 喉塞 | 分数: 0.948
  3. 凹端内六角紧定螺钉.ccm
     类型: 螺钉 | 分数: 0.935
```

### LLM增强模式
```
🔍 搜索: '三通管件'
  1. AGES-50.ccm
     类型: 四通 | 分数: 1.000
     💡 这是一个四通管件，可以用作三通（堵住一个口）
     相关: 三通, 弯头, 直通接头

🔍 搜索: 'M6螺丝'
  1. 内六角机螺钉 内六角平头带垫螺栓[PSHFP-M6-25-A].ccm
     类型: 螺钉 | 分数: 0.95
     💡 这是M6规格的内六角螺钉，螺纹长度25mm
     相关: 外六角螺栓, 沉头螺钉
  2. 凹端内六角紧定螺钉[PSEP-6-16-A].ccm
     类型: 螺钉 | 分数: 0.82
     💡 M6规格紧定螺钉，长度16mm
```

### 对话式搜索
```
👤 用户: 我需要找管道连接件
🤖 助手: 您需要什么类型的连接件？我们有：
   - 三通/四通（分流用）
   - 弯头（转向用）
   - 法兰（可拆卸连接）
   
   请告诉我您的具体需求或规格。

👤 用户: 有没有能分成三个方向的？
🤖 助手: 为您找到以下适合的零件：
   1. AGES-50（四通，可堵住一口作三通用）
   
   💡 建议：如果您需要精确的三通，可能需要补充库中的零件。
   您是否需要了解这个四通的详细规格？
```

---

**作者**: Claude  
**日期**: 2025-12-05  
**版本**: 2.0.0 (LLM增强版)
